/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.security.resources;

import com.google.common.collect.ImmutableSet;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.OpenSearchStatusException;
import org.opensearch.common.Nullable;
import org.opensearch.common.inject.Inject;
import org.opensearch.common.util.concurrent.ThreadContext;
import org.opensearch.core.action.ActionListener;
import org.opensearch.core.rest.RestStatus;
import org.opensearch.security.auth.UserSubjectImpl;
import org.opensearch.security.configuration.AdminDNs;
import org.opensearch.security.privileges.PrivilegesEvaluationContext;
import org.opensearch.security.privileges.PrivilegesEvaluator;
import org.opensearch.security.privileges.actionlevel.RoleBasedActionPrivileges;
import org.opensearch.security.resources.ResourcePluginInfo;
import org.opensearch.security.resources.ResourceSharingIndexHandler;
import org.opensearch.security.resources.SharingRecord;
import org.opensearch.security.securityconf.FlattenedActionGroups;
import org.opensearch.security.spi.resources.sharing.Recipient;
import org.opensearch.security.spi.resources.sharing.ResourceSharing;
import org.opensearch.security.spi.resources.sharing.ShareWith;
import org.opensearch.security.support.WildcardMatcher;
import org.opensearch.security.user.User;
import org.opensearch.threadpool.ThreadPool;
import reactor.util.annotation.NonNull;

public class ResourceAccessHandler {
    private static final Logger LOGGER = LogManager.getLogger(ResourceAccessHandler.class);
    private final ThreadContext threadContext;
    private final ResourceSharingIndexHandler resourceSharingIndexHandler;
    private final AdminDNs adminDNs;
    private final PrivilegesEvaluator privilegesEvaluator;
    private final ResourcePluginInfo resourcePluginInfo;

    @Inject
    public ResourceAccessHandler(ThreadPool threadPool, ResourceSharingIndexHandler resourceSharingIndexHandler, AdminDNs adminDns, PrivilegesEvaluator evaluator, ResourcePluginInfo resourcePluginInfo) {
        this.threadContext = threadPool.getThreadContext();
        this.resourceSharingIndexHandler = resourceSharingIndexHandler;
        this.adminDNs = adminDns;
        this.privilegesEvaluator = evaluator;
        this.resourcePluginInfo = resourcePluginInfo;
    }

    public void getOwnAndSharedResourceIdsForCurrentUser(@NonNull String resourceIndex, ActionListener<Set<String>> listener) {
        User user;
        UserSubjectImpl userSub = (UserSubjectImpl)this.threadContext.getPersistent("_opendistro_security_authenticated_user");
        User user2 = user = userSub == null ? null : userSub.getUser();
        if (user == null) {
            LOGGER.warn("No authenticated user; returning empty set of ids");
            listener.onResponse(Collections.emptySet());
            return;
        }
        if (this.adminDNs.isAdmin(user)) {
            this.loadAllResourceIds(resourceIndex, (ActionListener<Set<String>>)ActionListener.wrap(arg_0 -> listener.onResponse(arg_0), arg_0 -> listener.onFailure(arg_0)));
            return;
        }
        Set<String> flatPrincipals = this.getFlatPrincipals(user);
        this.resourceSharingIndexHandler.fetchAccessibleResourceIds(resourceIndex, flatPrincipals, listener);
    }

    public void getResourceSharingInfoForCurrentUser(@NonNull String resourceIndex, ActionListener<Set<SharingRecord>> listener) {
        User user;
        UserSubjectImpl userSub = (UserSubjectImpl)this.threadContext.getPersistent("_opendistro_security_authenticated_user");
        User user2 = user = userSub == null ? null : userSub.getUser();
        if (user == null) {
            LOGGER.warn("No authenticated user; returning empty set of resource-sharing records");
            listener.onResponse(Collections.emptySet());
            return;
        }
        if (this.adminDNs.isAdmin(user)) {
            this.loadAllResourceSharingRecords(resourceIndex, (ActionListener<Set<SharingRecord>>)ActionListener.wrap(arg_0 -> listener.onResponse(arg_0), arg_0 -> listener.onFailure(arg_0)));
            return;
        }
        Set<String> flatPrincipals = this.getFlatPrincipals(user);
        this.resourceSharingIndexHandler.fetchAccessibleResourceSharingRecords(resourceIndex, user, flatPrincipals, listener);
    }

    public void hasPermission(@NonNull String resourceId, @NonNull String resourceIndex, @NonNull String action, PrivilegesEvaluationContext context, ActionListener<Boolean> listener) {
        User user;
        UserSubjectImpl userSubject = (UserSubjectImpl)this.threadContext.getPersistent("_opendistro_security_authenticated_user");
        User user2 = user = userSubject == null ? null : userSubject.getUser();
        if (user == null) {
            LOGGER.warn("No authenticated user found. Access to resource {} is not authorized.", (Object)resourceId);
            listener.onResponse((Object)false);
            return;
        }
        LOGGER.info("Checking if user '{}' has permission to resource '{}'", (Object)user.getName(), (Object)resourceId);
        if (this.adminDNs.isAdmin(user)) {
            LOGGER.debug("User '{}' is admin, automatically granted permission on '{}'", (Object)user.getName(), (Object)resourceId);
            listener.onResponse((Object)true);
            return;
        }
        PrivilegesEvaluationContext effectiveContext = context != null ? context : this.privilegesEvaluator.createContext(user, action);
        HashSet<String> userRoles = new HashSet<String>((Collection<String>)user.getSecurityRoles());
        HashSet<String> userBackendRoles = new HashSet<String>((Collection<String>)user.getRoles());
        if (!(effectiveContext.getActionPrivileges() instanceof RoleBasedActionPrivileges)) {
            LOGGER.debug("Plugin/Token access to resources is currently not supported. {} is not authorized to access resource {}.", (Object)user.getName(), (Object)resourceId);
            listener.onResponse((Object)false);
            return;
        }
        this.resourceSharingIndexHandler.fetchSharingInfo(resourceIndex, resourceId, (ActionListener<ResourceSharing>)ActionListener.wrap(document -> {
            if (document == null) {
                LOGGER.warn("No sharing info found for '{}'. Action {} is not allowed.", (Object)resourceId, (Object)action);
                listener.onResponse((Object)false);
                return;
            }
            userRoles.add("*");
            userBackendRoles.add("*");
            if (document.isCreatedBy(user.getName())) {
                listener.onResponse((Object)true);
                return;
            }
            HashSet<String> accessLevels = new HashSet<String>();
            accessLevels.addAll(document.fetchAccessLevels(Recipient.USERS, Set.of(user.getName(), "*")));
            accessLevels.addAll(document.fetchAccessLevels(Recipient.ROLES, userRoles));
            accessLevels.addAll(document.fetchAccessLevels(Recipient.BACKEND_ROLES, userBackendRoles));
            if (accessLevels.isEmpty()) {
                listener.onResponse((Object)false);
                return;
            }
            String resourceType = this.resourcePluginInfo.typeByIndex(resourceIndex);
            if (resourceType == null) {
                LOGGER.debug("No resourceType mapping found for index '{}'; denying action {}", (Object)resourceIndex, (Object)action);
                listener.onResponse((Object)false);
                return;
            }
            FlattenedActionGroups agForType = this.resourcePluginInfo.flattenedForType(resourceType);
            ImmutableSet<String> allowedActions = agForType.resolve(accessLevels);
            WildcardMatcher matcher = WildcardMatcher.from(allowedActions);
            listener.onResponse((Object)matcher.test(action));
        }, e -> {
            LOGGER.error("Error while checking permission for user {} on resource {}: {}", (Object)user.getName(), (Object)resourceId, (Object)e.getMessage());
            listener.onFailure(e);
        }));
    }

    public void patchSharingInfo(@NonNull String resourceId, @NonNull String resourceIndex, @Nullable ShareWith add, @Nullable ShareWith revoke, ActionListener<ResourceSharing> listener) {
        User user;
        UserSubjectImpl userSubject = (UserSubjectImpl)this.threadContext.getPersistent("_opendistro_security_authenticated_user");
        User user2 = user = userSubject == null ? null : userSubject.getUser();
        if (user == null) {
            LOGGER.warn("No authenticated user found. Failed to patch resource sharing info {}", (Object)resourceId);
            listener.onFailure((Exception)new OpenSearchStatusException("No authenticated user found. Failed to patch resource sharing info " + resourceId, RestStatus.UNAUTHORIZED, new Object[0]));
            return;
        }
        LOGGER.debug("User {} is updating sharing info for resource {} in index {} with add: {}, revoke: {} ", (Object)user.getName(), (Object)resourceId, (Object)resourceIndex, (Object)add, (Object)revoke);
        this.resourceSharingIndexHandler.patchSharingInfo(resourceId, resourceIndex, add, revoke, (ActionListener<ResourceSharing>)ActionListener.wrap(sharingInfo -> {
            LOGGER.debug("Successfully patched sharing info for resource {} with add: {}, revoke: {}", (Object)resourceId, (Object)add, (Object)revoke);
            listener.onResponse(sharingInfo);
        }, e -> {
            LOGGER.error("Failed to patched sharing info for resource {} with add: {}, revoke: {} : {}", (Object)resourceId, (Object)add, (Object)revoke, (Object)e.getMessage());
            listener.onFailure(e);
        }));
    }

    public void getSharingInfo(@NonNull String resourceId, @NonNull String resourceIndex, ActionListener<ResourceSharing> listener) {
        User user;
        UserSubjectImpl userSubject = (UserSubjectImpl)this.threadContext.getPersistent("_opendistro_security_authenticated_user");
        User user2 = user = userSubject == null ? null : userSubject.getUser();
        if (user == null) {
            LOGGER.warn("No authenticated user found. Failed to fetch resource sharing info {}", (Object)resourceId);
            listener.onFailure((Exception)new OpenSearchStatusException("No authenticated user found. Failed to fetch resource sharing info " + resourceId, RestStatus.UNAUTHORIZED, new Object[0]));
            return;
        }
        LOGGER.debug("User {} is fetching sharing info for resource {} in index {}", (Object)user.getName(), (Object)resourceId, (Object)resourceIndex);
        this.resourceSharingIndexHandler.fetchSharingInfo(resourceIndex, resourceId, (ActionListener<ResourceSharing>)ActionListener.wrap(sharingInfo -> {
            LOGGER.debug("Successfully fetched sharing info for resource {} in index {}", (Object)resourceId, (Object)resourceIndex);
            listener.onResponse(sharingInfo);
        }, e -> {
            LOGGER.error("Failed to fetched sharing info for resource {} in index {}: {}", (Object)resourceId, (Object)resourceIndex, (Object)e.getMessage());
            listener.onFailure(e);
        }));
    }

    public void share(@NonNull String resourceId, @NonNull String resourceIndex, @NonNull ShareWith target, ActionListener<ResourceSharing> listener) {
        User user;
        UserSubjectImpl userSubject = (UserSubjectImpl)this.threadContext.getPersistent("_opendistro_security_authenticated_user");
        User user2 = user = userSubject == null ? null : userSubject.getUser();
        if (user == null) {
            LOGGER.warn("No authenticated user found. Failed to share resource {}", (Object)resourceId);
            listener.onFailure((Exception)new OpenSearchStatusException("No authenticated user found. Failed to share resource " + resourceId, RestStatus.UNAUTHORIZED, new Object[0]));
            return;
        }
        LOGGER.debug("Sharing resource {} created by {} with {}", (Object)resourceId, (Object)user.getName(), (Object)target.toString());
        this.resourceSharingIndexHandler.share(resourceId, resourceIndex, target, (ActionListener<ResourceSharing>)ActionListener.wrap(sharingInfo -> {
            LOGGER.debug("Successfully shared resource {} with {}", (Object)resourceId, (Object)target.toString());
            listener.onResponse(sharingInfo);
        }, e -> {
            LOGGER.error("Failed to share resource {} with {}: {}", (Object)resourceId, (Object)target.toString(), (Object)e.getMessage());
            listener.onFailure(e);
        }));
    }

    public void revoke(@NonNull String resourceId, @NonNull String resourceIndex, @NonNull ShareWith target, ActionListener<ResourceSharing> listener) {
        User user;
        UserSubjectImpl userSubject = (UserSubjectImpl)this.threadContext.getPersistent("_opendistro_security_authenticated_user");
        User user2 = user = userSubject == null ? null : userSubject.getUser();
        if (user == null) {
            LOGGER.warn("No authenticated user found. Failed to revoke access to resource {}", (Object)resourceId);
            listener.onFailure((Exception)new OpenSearchStatusException("No authenticated user found. Failed to revoke access to resource {}" + resourceId, RestStatus.UNAUTHORIZED, new Object[0]));
            return;
        }
        LOGGER.debug("User {} revoking access to resource {} for {}.", (Object)user.getName(), (Object)resourceId, (Object)target);
        this.resourceSharingIndexHandler.revoke(resourceId, resourceIndex, target, (ActionListener<ResourceSharing>)ActionListener.wrap(arg_0 -> listener.onResponse(arg_0), exception -> {
            LOGGER.error("Failed to revoke access to resource {} in index {}: {}", (Object)resourceId, (Object)resourceIndex, (Object)exception.getMessage());
            listener.onFailure(exception);
        }));
    }

    private void loadAllResourceIds(String resourceIndex, ActionListener<Set<String>> listener) {
        this.resourceSharingIndexHandler.fetchAllResourceIds(resourceIndex, listener);
    }

    private void loadAllResourceSharingRecords(String resourceIndex, ActionListener<Set<SharingRecord>> listener) {
        this.resourceSharingIndexHandler.fetchAllResourceSharingRecords(resourceIndex, listener);
    }

    private Set<String> getFlatPrincipals(User user) {
        HashSet<String> users = new HashSet<String>();
        users.add(user.getName());
        users.add("*");
        return Stream.concat(users.stream().map(u -> "user:" + u), Stream.concat(user.getSecurityRoles().stream().map(r -> "role:" + r), user.getRoles().stream().map(b -> "backend:" + b))).collect(Collectors.toSet());
    }
}

