/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"
#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_struct.h"

#include "lwpmudrv.h"
#include "utility.h"
#include "control.h"
#include "output.h"
#include "perfver4.h"
#include "ecb_iterators.h"
#include "pebs.h"
#include "apic.h"


extern U64                 *read_counter_info;
extern DRV_CONFIG                 drv_cfg;
extern EMON_BUFFER_DRIVER_HELPER  emon_buffer_driver_helper;

typedef struct SADDR_S {
    S64 addr:PERFVER4_LBR_DATA_BITS;
} SADDR;

#define SADDR_addr(x)                  (x).addr
#define MSR_ENERGY_MULTIPLIER          0x606        // Energy Multiplier MSR

#define IS_FIXED_CTR_ENABLED(ia32_perf_global_ctrl_reg_val)                          \
                            ((ia32_perf_global_ctrl_reg_val) & 0x700000000ULL)
#define IS_FOUR_FIXED_CTR_ENABLED(ia32_perf_global_ctrl_reg_val)                     \
                            ((ia32_perf_global_ctrl_reg_val) & 0xF00000000ULL)
#define IS_PMC_PEBS_ENABLED_GP(ia32_perf_global_ctrl_reg_val, ia32_pebs_enable_reg_val) \
                              (((ia32_perf_global_ctrl_reg_val) & 0xfULL) == ((ia32_pebs_enable_reg_val) & 0xfULL))
#define IS_PMC_PEBS_ENABLED_FP_AND_GP(ia32_perf_global_ctrl_reg_val, ia32_pebs_enable_reg_val) \
                                 (((ia32_perf_global_ctrl_reg_val) & 0xf000000ffULL) == ((ia32_pebs_enable_reg_val) & 0xf000000ffULL))

#define DISABLE_FRZ_ON_PMI(ia32_debug_ctrl_reg_val)                                  \
                            (0xefff & (ia32_debug_ctrl_reg_val))
/* ------------------------------------------------------------------------- */
/*!
 * @fn void perfver4_Write_PMU(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Initial set up of the PMU registers
 *
 * <I>Special Notes</I>
 *         Initial write of PMU registers.
 *         Walk through the enties and write the value of the register accordingly.
 *         Assumption:  For CCCR registers the enable bit is set to value 0.
 *         When current_group = 0, then this is the first time this routine is called,
 *         initialize the locks and set up EM tables.
 */
static VOID
perfver4_Write_PMU (
    VOID  *param
)
{
    U32            this_cpu      = CONTROL_THIS_CPU();
    CPU_STATE      pcpu          = &pcb[this_cpu];
    U32            dev_idx       = core_to_dev_map[this_cpu];
    U32            cur_grp       = CPU_STATE_current_group(pcpu);
    ECB            pecb          = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    DEV_CONFIG     pcfg          = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    EVENT_CONFIG   ec            = LWPMU_DEVICE_ec(&devices[dev_idx]);
    DISPATCH       dispatch      = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    if (!pecb) {
        return;
    }

    if (CPU_STATE_current_group(pcpu) == 0) {
        if (EVENT_CONFIG_mode(ec) != EM_DISABLED) {
            U32            index;
            U32            st_index;
            U32            j;

            /* Save all the initialization values away into an array for Event Multiplexing. */
            for (j = 0; j < EVENT_CONFIG_num_groups(ec); j++) {
                CPU_STATE_current_group(pcpu) = j;
                st_index   = CPU_STATE_current_group(pcpu) * EVENT_CONFIG_max_gp_events(ec);
                FOR_EACH_DATA_GP_REG(pecb, i) {
                    index = st_index + (ECB_entries_reg_id(pecb,i) - IA32_PMC0);
                    CPU_STATE_em_tables(pcpu)[index] = ECB_entries_reg_value(pecb,i);
                } END_FOR_EACH_DATA_GP_REG;
            }
            /* Reset the current group to the very first one. */
            CPU_STATE_current_group(pcpu) = this_cpu % EVENT_CONFIG_num_groups(ec);
        }
    }

    if (dispatch->hw_errata) {
        dispatch->hw_errata();
    }

    /* Clear outstanding frozen bits */
    SYS_Write_MSR(IA32_PERF_GLOBAL_OVF_CTRL, PERFVER4_FROZEN_BIT_MASK);

    FOR_EACH_REG_ENTRY(pecb, i) {
#if __FreeBSD_version < 1101511 || defined(DISABLE_PEBS)
        // FIXME: convert all precise events to normal event temporarily for meltdown patch
        if (!DRV_CONFIG_emon_mode(drv_cfg)
            && ECB_entries_reg_id(pecb,i) >= IA32_PERFEVTSEL0
            && ECB_entries_reg_id(pecb,i) <= IA32_PERFEVTSEL1 + 6) {
            ECB_entries_reg_value(pecb,i) |= (1 << 20);
        }
#endif
        /*
         * Writing the GLOBAL Control register enables the PMU to start counting.
         * So write 0 into the register to prevent any counting from starting.
         */
        if (ECB_entries_reg_id(pecb,i) == IA32_PERF_GLOBAL_CTRL) {
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
            continue;
        }
        /*
         *  PEBS is enabled for this collection session
         */
        if (ECB_entries_reg_id(pecb,i) == IA32_PEBS_ENABLE &&
            ECB_entries_reg_value(pecb,i)) {
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
            continue;
        }

        if (DEV_CONFIG_pebs_mode(pcfg)
            && (ECB_entries_precise_get(pecb, i) == 1)) {
            PEBS_Reset_Counter(this_cpu,
                               i,
                               PMU_OPERATION_ALL_REG);
        }

        SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));

    } END_FOR_EACH_REG_ENTRY;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void perfver4_Disable_PMU(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Zero out the global control register.  This automatically disables the PMU counters.
 *
 */
static VOID
perfver4_Disable_PMU (
    PVOID  param
)
{
    U32         this_cpu = CONTROL_THIS_CPU();
    CPU_STATE   pcpu     = &pcb[this_cpu];
    U32         dev_idx  = core_to_dev_map[this_cpu];
    U32         cur_grp  = CPU_STATE_current_group(pcpu);
    ECB         pecb     = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    DEV_CONFIG  pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!pecb) {
        // no programming for this device for this group
        return;
    }

    if (GET_DRIVER_STATE() != DRV_STATE_RUNNING) {
        SEP_DRV_LOG_TRACE("driver state = %d.", GET_DRIVER_STATE());
        SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL, 0LL);
        if (DEV_CONFIG_pebs_mode(pcfg)) {
            SYS_Write_MSR(IA32_PEBS_ENABLE, 0LL);
        }

        APIC_Disable_PMI();
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void perfver4_Enable_PMU(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Set the enable bit for all the Control registers
 *
 */
static VOID
perfver4_Enable_PMU (
    PVOID   param
)
{
    /*
     * Get the value from the event block
     *   0 == location of the global control reg for this block.
     *   Generalize this location awareness when possible
     */
    U32         this_cpu = CONTROL_THIS_CPU();
    CPU_STATE   pcpu     = &pcb[this_cpu];
    U32         dev_idx  = core_to_dev_map[this_cpu];
    U32         cur_grp  = CPU_STATE_current_group(pcpu);
    ECB         pecb     = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    DEV_CONFIG  pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!pecb) {
        // no programming for this device for this group
        return;
    }

    if (GET_DRIVER_STATE() == DRV_STATE_RUNNING) {
        APIC_Enable_Pmi();

        /* Clear outstanding frozen bits */
        SYS_Write_MSR(IA32_PERF_GLOBAL_OVF_CTRL, PERFVER4_FROZEN_BIT_MASK);

        if (CPU_STATE_reset_mask(pcpu)) {
            SEP_DRV_LOG_TRACE("Overflow reset mask %llx.", CPU_STATE_reset_mask(pcpu));
            SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL,  ECB_entries_reg_value(pecb,0));
            // Reinitialize the global overflow control register
            SYS_Write_MSR(IA32_DEBUG_CTRL, ECB_entries_reg_value(pecb,3));
            CPU_STATE_reset_mask(pcpu) = 0LL;
        }
        if (CPU_STATE_group_swap(pcpu)) {
            CPU_STATE_group_swap(pcpu) = 0;
            SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL, ECB_entries_reg_value(pecb,0));
            if (DEV_CONFIG_pebs_mode(pcfg)) {
                SYS_Write_MSR(IA32_PEBS_ENABLE, ECB_entries_reg_value(pecb,2));
            }
            SYS_Write_MSR(IA32_DEBUG_CTRL, ECB_entries_reg_value(pecb,3));

        }
    }
    SEP_DRV_LOG_TRACE("Reenabled PMU with value 0x%llx.", ECB_entries_reg_value(pecb,0));

    return;
}


#define FIXED_CTR3_BIT_INDEX       35
/* ------------------------------------------------------------------------- */
/*!
 * @fn perfver4_Read_PMU_Data(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Read all the data MSR's into a buffer.  Called by the interrupt handler.
 *
 */
static void
perfver4_Read_PMU_Data (
    PVOID   param,
    U32     dev_idx
)
{
    U32         j;
    U64        *buffer    = (U64 *)param;
    U32         this_cpu;
    CPU_STATE   pcpu;
    ECB         pecb;
    U32         cur_grp;
    U32         index;
    DEV_CONFIG  pcfg;
    DISPATCH    dispatch;

    preempt_disable();
    this_cpu  = CONTROL_THIS_CPU();
    preempt_enable();
    pcpu      = &pcb[this_cpu];
    cur_grp   = CPU_STATE_current_group(pcpu);
    pecb      = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    index     = 0;
    pcfg      = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    dispatch  = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    if (!pecb) {
        return;
    }

    SEP_DRV_LOG_TRACE("PMU control_data %p, buffer %p.", LWPMU_DEVICE_PMU_register_data(&devices[dev_idx]), buffer);
    LFENCE_SERIALIZE();
    FOR_EACH_DATA_REG(pecb,i) {

        j = EMON_BUFFER_CORE_EVENT_OFFSET(EMON_BUFFER_DRIVER_HELPER_core_index_to_thread_offset_map(emon_buffer_driver_helper)[this_cpu],
                                          ECB_entries_core_event_id(pecb,i));
        buffer[j] = SYS_Read_PMC(ECB_entries_reg_id(pecb,i), ECB_entries_fixed_reg_get(pecb, i));
        SEP_DRV_LOG_TRACE("j=%u, value=%llu, cpu=%u, event_id=%u", j, buffer[j], this_cpu, ECB_entries_core_event_id(pecb,i));

        /*
         * Read hardware perf_metrics if appropriate fixed counter is specified
         * for collection
         */
        if (ECB_entries_fixed_reg_get(pecb, i)) {
            index = ECB_entries_reg_id(pecb, i) - IA32_FIXED_CTR0 + 0x20;
            if (DEV_CONFIG_enable_perf_metrics(pcfg) && index == FIXED_CTR3_BIT_INDEX) {
                j = EMON_BUFFER_CORE_EVENT_OFFSET(EMON_BUFFER_DRIVER_HELPER_core_index_to_thread_offset_map(emon_buffer_driver_helper)[this_cpu],
                                                 EMON_BUFFER_DRIVER_HELPER_core_num_events(emon_buffer_driver_helper));
                dispatch->read_metrics((PVOID)(buffer+j));
            }
        }

    } END_FOR_EACH_DATA_REG;
    LFENCE_SERIALIZE();

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void perfver4_Check_Overflow(masks)
 *
 * @param    masks    the mask structure to populate
 *
 * @return   None     No return needed
 *
 * @brief  Called by the data processing method to figure out which registers have overflowed.
 *
 */
static void
perfver4_Check_Overflow (
    DRV_MASKS    masks
)
{
    U32              index;
    U64              overflow_status     = 0;
    U32              this_cpu            = CONTROL_THIS_CPU();
    BUFFER_DESC      bd                  = &cpu_buf[this_cpu];
    CPU_STATE        pcpu                = &pcb[this_cpu];
    U32              dev_idx             = core_to_dev_map[this_cpu];
    U32              cur_grp             = CPU_STATE_current_group(pcpu);
    ECB              pecb                = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    DEV_CONFIG       pcfg                = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH         dispatch            = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
    U64              overflow_status_clr = 0;
    DRV_EVENT_MASK_NODE event_flag;

    if (!pecb) {
        return;
    }

    // initialize masks
    DRV_MASKS_masks_num(masks) = 0;

    overflow_status = SYS_Read_MSR(IA32_PERF_GLOBAL_STATUS);

    if (DEV_CONFIG_pebs_mode(pcfg)) {
        overflow_status = PEBS_Overflowed (this_cpu, overflow_status);
    }
    overflow_status_clr = overflow_status;

    if (dispatch->check_overflow_gp_errata) {
        overflow_status = dispatch->check_overflow_gp_errata(pecb,  &overflow_status_clr);
    }
    SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, status 0x%llx .", this_cpu, overflow_status);
    index                        = 0;
    BUFFER_DESC_sample_count(bd) = 0;
    FOR_EACH_DATA_REG(pecb, i) {
        if (ECB_entries_fixed_reg_get(pecb, i)) {
            index = ECB_entries_reg_id(pecb, i) - IA32_FIXED_CTR0 + 0x20;
            if (dispatch->check_overflow_errata) {
                overflow_status = dispatch->check_overflow_errata(pecb, i, overflow_status);
            }
        }
        else if (ECB_entries_is_gp_reg_get(pecb, i)) {
            index = ECB_entries_reg_id(pecb, i) - IA32_PMC0;
        }
        else {
            continue;
        }
        if (overflow_status & ((U64)1 << index)) {
            SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, index %d.", this_cpu, index);
            SEP_DRV_LOG_TRACE("register 0x%x --- val 0%llx.",
                            ECB_entries_reg_id(pecb,i),
                            SYS_Read_MSR(ECB_entries_reg_id(pecb,i)));
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));

            DRV_EVENT_MASK_bitFields1(&event_flag) = (U8) 0;
            if (ECB_entries_precise_get(pecb, i)) {
                DRV_EVENT_MASK_precise(&event_flag) = 1;
            }
            if (ECB_entries_lbr_value_get(pecb, i)) {
                DRV_EVENT_MASK_lbr_capture(&event_flag) = 1;
            }
            if (ECB_entries_uncore_get(pecb, i)) {
                DRV_EVENT_MASK_uncore_capture(&event_flag) = 1;
            }
            if (ECB_entries_branch_evt_get(pecb, i)) {
                DRV_EVENT_MASK_branch(&event_flag) = 1;
            }
            if (ECB_entries_em_trigger_get(pecb,i)) {
                DRV_EVENT_MASK_trigger(&event_flag) = 1;
            }
            if (ECB_entries_collect_on_ctx_sw_get(pecb, i)) {
                DRV_EVENT_MASK_collect_on_ctx_sw(&event_flag) = 1;
            }

            if (DRV_MASKS_masks_num(masks) < MAX_OVERFLOW_EVENTS) {
                DRV_EVENT_MASK_bitFields1(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = DRV_EVENT_MASK_bitFields1(&event_flag);
                DRV_EVENT_MASK_event_idx(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_event_id_index(pecb, i);
                DRV_EVENT_MASK_desc_id(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_desc_id(pecb, i);
                DRV_MASKS_masks_num(masks)++;
            }
            else {
                SEP_DRV_LOG_ERROR("The array for event masks is full.");
            }

            SEP_DRV_LOG_TRACE("overflow -- 0x%llx, index 0x%llx.", overflow_status, (U64)1 << index);
            SEP_DRV_LOG_TRACE("slot# %d, reg_id 0x%x, index %d.",
                            i, ECB_entries_reg_id(pecb, i), index);
            if (ECB_entries_event_id_index(pecb, i) == CPU_STATE_trigger_event_num(pcpu)) {
                CPU_STATE_trigger_count(pcpu)--;
            }
        }
    } END_FOR_EACH_DATA_REG;

    CPU_STATE_reset_mask(pcpu) = overflow_status_clr;
    /* Clear outstanding overflow bits */
    SYS_Write_MSR(IA32_PERF_GLOBAL_OVF_CTRL, overflow_status_clr & PERFVER4_OVERFLOW_BIT_MASK_HT_ON);

    SEP_DRV_LOG_TRACE("Check Overflow completed %d.", this_cpu);
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn perfver4_Swap_Group(restart)
 *
 * @param    restart    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Perform the mechanics of swapping the event groups for event mux operations
 *
 * <I>Special Notes</I>
 *         Swap function for event multiplexing.
 *         Freeze the counting.
 *         Swap the groups.
 *         Enable the counting.
 *         Reset the event trigger count
 *
 */
static VOID
perfver4_Swap_Group (
    DRV_BOOL  restart
)
{
    U32            index;
    U32            next_group;
    U32            st_index;
    U32            this_cpu      = CONTROL_THIS_CPU();
    CPU_STATE      pcpu          = &pcb[this_cpu];
    U32            dev_idx       = core_to_dev_map[this_cpu];
    DEV_CONFIG     pcfg          = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    EVENT_CONFIG   ec            = LWPMU_DEVICE_ec(&devices[dev_idx]);
    DISPATCH       dispatch      = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg)) {
        return;
    }

    st_index   = CPU_STATE_current_group(pcpu) * EVENT_CONFIG_max_gp_events(ec);
    next_group = (CPU_STATE_current_group(pcpu) + 1);
    if (next_group >= EVENT_CONFIG_num_groups(ec)) {
        next_group = 0;
    }

    SEP_DRV_LOG_TRACE("current group : 0x%x.", CPU_STATE_current_group(pcpu));
    SEP_DRV_LOG_TRACE("next group : 0x%x.", next_group);

    // Save the counters for the current group
    FOR_EACH_DATA_GP_REG(pecb, i) {
        if (ECB_entries_ebc_sampling_evt_get(pecb, i) ||
            !DRV_CONFIG_event_based_counts(drv_cfg)) {
            index = st_index + (ECB_entries_reg_id(pecb, i) - IA32_PMC0);
            CPU_STATE_em_tables(pcpu)[index] = SYS_Read_MSR(ECB_entries_reg_id(pecb,i));
            SEP_DRV_LOG_TRACE("Saved value for cur grp %u, reg 0x%x : 0x%llx ",
                               CPU_STATE_current_group(pcpu),
                               ECB_entries_reg_id(pecb,i),
                               CPU_STATE_em_tables(pcpu)[index]);
        }
    } END_FOR_EACH_DATA_GP_REG;

    CPU_STATE_current_group(pcpu) = next_group;

    if (dispatch->hw_errata) {
        dispatch->hw_errata();
    }

    // First write the GP control registers (eventsel)
    FOR_EACH_CCCR_GP_REG(pecb, i) {
        SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));
    } END_FOR_EACH_CCCR_GP_REG;

    if (DRV_CONFIG_event_based_counts(drv_cfg)) {
        // In EBC mode, reset the counts for all events except for trigger event
        FOR_EACH_DATA_REG(pecb, i) {
            if (!ECB_entries_ebc_sampling_evt_get(pecb, i) &&
                (ECB_entries_event_id_index(pecb, i) != CPU_STATE_trigger_event_num(pcpu))) {
                SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
            }
        } END_FOR_EACH_DATA_REG;
    }
    // Then write the gp count registers
    st_index = CPU_STATE_current_group(pcpu) * EVENT_CONFIG_max_gp_events(ec);
    FOR_EACH_DATA_GP_REG(pecb, i) {
        if (ECB_entries_ebc_sampling_evt_get(pecb, i) ||
            !DRV_CONFIG_event_based_counts(drv_cfg)) {
            index = st_index + (ECB_entries_reg_id(pecb, i) - IA32_PMC0);
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), CPU_STATE_em_tables(pcpu)[index]);
            SEP_DRV_LOG_TRACE("Restore value for next grp %u, reg 0x%x : 0x%llx ",
                               next_group,
                               ECB_entries_reg_id(pecb,i),
                               CPU_STATE_em_tables(pcpu)[index]);
        }
    } END_FOR_EACH_DATA_GP_REG;

    FOR_EACH_ESCR_REG(pecb,i) {
        SYS_Write_MSR(ECB_entries_reg_id(pecb, i),ECB_entries_reg_value(pecb, i));
    } END_FOR_EACH_ESCR_REG;

    if (DEV_CONFIG_pebs_mode(pcfg) && DEV_CONFIG_pebs_record_num(pcfg)) {
        FOR_EACH_DATA_REG(pecb, i) {
            if (ECB_entries_precise_get(pecb, i) == 1) {
                PEBS_Reset_Counter(this_cpu,
                                   i,
                                   PMU_OPERATION_DATA_ALL);
            }
        } END_FOR_EACH_DATA_REG;
    }

    /*
     *  reset the em factor when a group is swapped
     */
    CPU_STATE_trigger_count(pcpu) = EVENT_CONFIG_em_factor(ec);

    /*
     * The enable routine needs to rewrite the control registers
     */
    CPU_STATE_reset_mask(pcpu) = 0LL;
    CPU_STATE_group_swap(pcpu) = 1;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn perfver4_Initialize(params)
 *
 * @param    params    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Initialize the PMU setting up for collection
 *
 * <I>Special Notes</I>
 *         Saves the relevant PMU state (minimal set of MSRs required
 *         to avoid conflicts with other Linux tools, such as Oprofile).
 *         This function should be called in parallel across all CPUs
 *         prior to the start of sampling, before PMU state is changed.
 *
 */
static VOID
perfver4_Initialize (
    VOID  *param
)
{
    U32        this_cpu = CONTROL_THIS_CPU();
    CPU_STATE  pcpu;
    U32        dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    SEP_DRV_LOG_TRACE("Inside perfver4_Initialize.");

    if (!DEV_CONFIG_num_events(pcfg)) {
        return;
    }

    if (pcb == NULL) {
        return;
    }

    pcpu  = &pcb[this_cpu];
    CPU_STATE_pmu_state(pcpu) = pmu_state + (this_cpu * 3);
    if (CPU_STATE_pmu_state(pcpu) == NULL) {
        SEP_DRV_LOG_WARNING("Unable to save PMU state on CPU %d.",this_cpu);
        return;
    }

    // save the original PMU state on this CPU (NOTE: must only be called ONCE per collection)
    CPU_STATE_pmu_state(pcpu)[0] = SYS_Read_MSR(IA32_DEBUG_CTRL);
    CPU_STATE_pmu_state(pcpu)[1] = SYS_Read_MSR(IA32_PERF_GLOBAL_CTRL);
    CPU_STATE_pmu_state(pcpu)[2] = SYS_Read_MSR(IA32_FIXED_CTRL);

    if (DRV_CONFIG_ds_area_available(drv_cfg) && DEV_CONFIG_pebs_mode(pcfg)) {
        SYS_Write_MSR(IA32_PEBS_ENABLE, 0LL);
    }

    SEP_DRV_LOG_TRACE("Saving PMU state on CPU %d :", this_cpu);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_DEBUG_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[0]);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_PERF_GLOBAL_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[1]);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_FIXED_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[2]);

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn perfver4_Destroy(params)
 *
 * @param    params    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Reset the PMU setting up after collection
 *
 * <I>Special Notes</I>
 *         Restores the previously saved PMU state done in pmv_v4_Initialize.
 *         This function should be called in parallel across all CPUs
 *         after sampling collection ends/terminates.
 *
 */
static VOID
perfver4_Destroy (
    VOID *param
)
{
    U32        this_cpu;
    CPU_STATE  pcpu;

    SEP_DRV_LOG_TRACE("Inside perfver4_Destroy.");

    if (pcb == NULL) {
        return;
    }

    preempt_disable();
    this_cpu = CONTROL_THIS_CPU();
    preempt_enable();
    pcpu = &pcb[this_cpu];

    if (CPU_STATE_pmu_state(pcpu) == NULL) {
        SEP_DRV_LOG_WARNING("Unable to restore PMU state on CPU %d.",this_cpu);
        return;
    }

    SEP_DRV_LOG_TRACE("Restoring PMU state on CPU %d :", this_cpu);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_DEBUG_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[0]);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_PERF_GLOBAL_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[1]);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_FIXED_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[2]);

    // restore the previously saved PMU state
    // (NOTE: assumes this is only called ONCE per collection)
    SYS_Write_MSR(IA32_DEBUG_CTRL, CPU_STATE_pmu_state(pcpu)[0]);
    SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL, CPU_STATE_pmu_state(pcpu)[1]);
    SYS_Write_MSR(IA32_FIXED_CTRL, CPU_STATE_pmu_state(pcpu)[2]);

    CPU_STATE_pmu_state(pcpu) = NULL;

    return;
}

/*
 * @fn perfver4_Read_LBRs(buffer)
 *
 * @param   IN buffer - pointer to the buffer to write the data into
 * @return  Last branch source IP address
 *
 * @brief   Read all the LBR registers into the buffer provided and return
 *
 */
static U64
perfver4_Read_LBRs (
    VOID   *buffer
)
{
    U32   i, count    = 0;
    U64  *lbr_buf     = NULL;
    U64   value       = 0;
    U64   tos_ip_addr = 0;
    U64   tos_ptr     = 0;
    SADDR saddr;
    U32   pairs       = 0;
    U32   this_cpu    = CONTROL_THIS_CPU();
    U32   dev_idx     = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg   = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    LBR   lbr         = LWPMU_DEVICE_lbr(&devices[dev_idx]);

    if (lbr == NULL) {
        return 0;
    }
    if (buffer && DEV_CONFIG_store_lbrs(pcfg)) {
        lbr_buf = (U64 *)buffer;
    }
    SEP_DRV_LOG_TRACE("Inside perfver4_Read_LBRs.");
    if (LBR_num_entries(lbr) > 0) {
        pairs = (LBR_num_entries(lbr) - 1)/3;
    }
    for (i = 0; i < LBR_num_entries(lbr); i++) {
        if (i == 0 && DEV_CONFIG_enable_arch_lbrs(pcfg)) {
            value = DEV_CONFIG_num_lbr_entries(pcfg)-1;
        }
        else if (LBR_entries_reg_id(lbr,i)) {
            value = SYS_Read_MSR(LBR_entries_reg_id(lbr,i));
        }
        else {
            continue;
        }
        if (buffer && DEV_CONFIG_store_lbrs(pcfg)) {
            *lbr_buf = value;
        }
        SEP_DRV_LOG_TRACE("perfver4_Read_LBRs %u, 0x%llx.", i, value);
        if (i == 0) {
            tos_ptr = value;
        }
        else {
            if (LBR_entries_etype(lbr, i) == LBR_ENTRY_FROM_IP) { // LBR from register
                if (tos_ptr == count) {
                    SADDR_addr(saddr) = value & PERFVER4_LBR_BITMASK;
                    tos_ip_addr = (U64) SADDR_addr(saddr); // Add signed extension
                    SEP_DRV_LOG_TRACE("tos_ip_addr %llu, 0x%llx.", tos_ptr, value);
                }
                count++;
            }
        }
        if (buffer && DEV_CONFIG_store_lbrs(pcfg)) {
            lbr_buf++;
        }
    }

    return tos_ip_addr;
}

/*
 * @fn perfver4_Clean_Up(param)
 *
 * @param   IN param - currently not used
 *
 * @brief   Clean up registers in ECB
 *
 */
static VOID
perfver4_Clean_Up (
    VOID   *param
)
{
    FOR_EACH_REG_ENTRY_IN_ALL_GRPS(pecb, i) {
        if (ECB_entries_clean_up_get(pecb,i)) {
            SEP_DRV_LOG_TRACE("clean up set --- RegId --- %x.", ECB_entries_reg_id(pecb,i));
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
        }
    } END_FOR_EACH_REG_ENTRY_IN_ALL_GRPS;

    /* Clear outstanding frozen bits */
    SYS_Write_MSR(IA32_PERF_GLOBAL_OVF_CTRL, PERFVER4_FROZEN_BIT_MASK);

    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn void perfver4_Check_Overflow_Htoff_Mode(masks)
 *
 * @param    masks    the mask structure to populate
 *
 * @return   None     No return needed
 *
 * @brief  Called by the data processing method to figure out which registers have overflowed.
 *
 */
static void
perfver4_Check_Overflow_Htoff_Mode (
    DRV_MASKS    masks
)
{
    U32              index;
    U64              value               = 0;
    U64              overflow_status     = 0;
    U32              this_cpu            = CONTROL_THIS_CPU();
    BUFFER_DESC      bd                  = &cpu_buf[this_cpu];
    CPU_STATE        pcpu                = &pcb[this_cpu];
    U32              dev_idx             = core_to_dev_map[this_cpu];
    U32              cur_grp             = CPU_STATE_current_group(pcpu);
    ECB              pecb                = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    DEV_CONFIG       pcfg                = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH         dispatch            = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
    U64              overflow_status_clr = 0;
    DRV_EVENT_MASK_NODE event_flag;

    SEP_DRV_LOG_TRACE("perfver4_Check_Overflow_Htoff_Mode.");

    if (!pecb) {
        return;
    }

    // initialize masks
    DRV_MASKS_masks_num(masks) = 0;

    overflow_status = SYS_Read_MSR(IA32_PERF_GLOBAL_STATUS);

    if (DEV_CONFIG_pebs_mode(pcfg)) {
        overflow_status = PEBS_Overflowed (this_cpu, overflow_status);
    }
    overflow_status_clr = overflow_status;
    SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, status 0x%llx .", this_cpu, overflow_status);
    index                        = 0;
    BUFFER_DESC_sample_count(bd) = 0;

    if (dispatch->check_overflow_gp_errata) {
        overflow_status = dispatch->check_overflow_gp_errata(pecb,  &overflow_status_clr);
    }

    FOR_EACH_DATA_REG(pecb, i) {
        if (ECB_entries_fixed_reg_get(pecb, i)) {
            index = ECB_entries_reg_id(pecb, i) - IA32_FIXED_CTR0 + 0x20;
        }
        else if (ECB_entries_is_gp_reg_get(pecb,i) && ECB_entries_reg_value(pecb,i) != 0) {
            index = ECB_entries_reg_id(pecb, i) - IA32_PMC0;
            if (ECB_entries_reg_id(pecb, i) >= IA32_PMC4 &&
                ECB_entries_reg_id(pecb, i) <= IA32_PMC7) {
                value = SYS_Read_MSR(ECB_entries_reg_id(pecb,i));
                if (value > 0 && value <= 0x100000000LL) {
                    overflow_status |= ((U64)1 << index);
                }
            }
        }
        else {
            continue;
        }
        if (overflow_status & ((U64)1 << index)) {
            SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, index %d.", this_cpu, index);
            SEP_DRV_LOG_TRACE("register 0x%x --- val 0%llx.",
                            ECB_entries_reg_id(pecb,i),
                            SYS_Read_MSR(ECB_entries_reg_id(pecb,i)));
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));

            DRV_EVENT_MASK_bitFields1(&event_flag) = (U8) 0;
            if (ECB_entries_precise_get(pecb, i)) {
                DRV_EVENT_MASK_precise(&event_flag) = 1;
            }
            if (ECB_entries_lbr_value_get(pecb, i)) {
                DRV_EVENT_MASK_lbr_capture(&event_flag) = 1;
            }
            if (ECB_entries_em_trigger_get(pecb,i)) {
                DRV_EVENT_MASK_trigger(&event_flag) = 1;
            }
            if (ECB_entries_collect_on_ctx_sw_get(pecb, i)) {
                DRV_EVENT_MASK_collect_on_ctx_sw(&event_flag) = 1;
            }

            if (DRV_MASKS_masks_num(masks) < MAX_OVERFLOW_EVENTS) {
                DRV_EVENT_MASK_bitFields1(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = DRV_EVENT_MASK_bitFields1(&event_flag);
                DRV_EVENT_MASK_event_idx(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_event_id_index(pecb, i);
                DRV_EVENT_MASK_desc_id(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_desc_id(pecb, i);
                DRV_MASKS_masks_num(masks)++;
            }
            else {
                SEP_DRV_LOG_ERROR("The array for event masks is full.");
            }

            SEP_DRV_LOG_TRACE("overflow -- 0x%llx, index 0x%llx.", overflow_status, (U64)1 << index);
            SEP_DRV_LOG_TRACE("slot# %d, reg_id 0x%x, index %d.",
                             i, ECB_entries_reg_id(pecb, i), index);
            if (ECB_entries_event_id_index(pecb, i) == CPU_STATE_trigger_event_num(pcpu)) {
                CPU_STATE_trigger_count(pcpu)--;
            }
        }
    } END_FOR_EACH_DATA_REG;

    CPU_STATE_reset_mask(pcpu) = overflow_status_clr;
    /* Clear outstanding overflow bits */
    SYS_Write_MSR(IA32_PERF_GLOBAL_OVF_CTRL, overflow_status_clr & PERFVER4_OVERFLOW_BIT_MASK_HT_OFF);

    SEP_DRV_LOG_TRACE("Check Overflow completed %d.", this_cpu);
}

#define MAX_COUNTER                0xFFFFFFFFFFFFLLU

/* ------------------------------------------------------------------------- */
/*!
 * @fn void perfver4_Check_Overflow_Nonht_Mode(masks)
 *
 * @param    masks    the mask structure to populate
 *
 * @return   None     No return needed
 *
 * @brief  Called by the data processing method to figure out which registers have overflowed.
 *         This method checks overflows on 4 fixed and 8 GP counters irrespective of HT on/off
 *         8 GP and 4 Fixed counters are available in SunnyCove/ICL
 *
 */
static VOID
perfver4_Check_Overflow_Nonht_Mode (
    DRV_MASKS    masks
)
{
    U32              index;
    U64              overflow_status     = 0;
    U32              this_cpu            = CONTROL_THIS_CPU();
    BUFFER_DESC      bd                  = &cpu_buf[this_cpu];
    CPU_STATE        pcpu                = &pcb[this_cpu];
    U32              dev_idx             = core_to_dev_map[this_cpu];
    U32              cur_grp             = CPU_STATE_current_group(pcpu);
    ECB              pecb                = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    DEV_CONFIG       pcfg                = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    U64              overflow_status_clr = 0;
    DRV_EVENT_MASK_NODE event_flag;


    if (!pecb) {
        SEP_DRV_LOG_TRACE("Early exit (!pecb).");
        return;
    }

    // initialize masks
    DRV_MASKS_masks_num(masks) = 0;

    overflow_status = SYS_Read_MSR(IA32_PERF_GLOBAL_STATUS);

    if (DEV_CONFIG_pebs_mode(pcfg)) {
        overflow_status = PEBS_Overflowed (this_cpu, overflow_status);
    }
    overflow_status_clr = overflow_status;
    SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, status 0x%llx.", this_cpu, overflow_status);
    index                        = 0;
    BUFFER_DESC_sample_count(bd) = 0;

    FOR_EACH_DATA_REG(pecb, i) {
        if (ECB_entries_fixed_reg_get(pecb, i)) {
            index = ECB_entries_reg_id(pecb, i) - IA32_FIXED_CTR0 + 0x20;
        }
        else if (ECB_entries_is_gp_reg_get(pecb,i) && ECB_entries_reg_value(pecb,i) != 0) {
            index = ECB_entries_reg_id(pecb, i) - IA32_PMC0;
        }
        else {
            continue;
        }
        if (overflow_status & ((U64)1 << index)) {
            SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, index %d.", this_cpu, index);
            SEP_DRV_LOG_TRACE("register 0x%x --- val 0%llx.",
                            ECB_entries_reg_id(pecb,i),
                            SYS_Read_MSR(ECB_entries_reg_id(pecb,i)));
            DRV_EVENT_MASK_bitFields1(&event_flag) = (U8) 0;
            if (DEV_CONFIG_enable_perf_metrics(pcfg) && index == FIXED_CTR3_BIT_INDEX) {
                // Writing positive SAV into data register before reading metrics
                SYS_Write_MSR(ECB_entries_reg_id(pecb,i),((~(ECB_entries_reg_value(pecb,i)) + 1) & MAX_COUNTER));
                DRV_EVENT_MASK_perf_metrics_capture(&event_flag) = 1;
            }
            else {
                 SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));
            }


            if (ECB_entries_precise_get(pecb, i)) {
                DRV_EVENT_MASK_precise(&event_flag) = 1;
            }
            if (ECB_entries_lbr_value_get(pecb, i)) {
                DRV_EVENT_MASK_lbr_capture(&event_flag) = 1;
            }
            if (ECB_entries_uncore_get(pecb, i)) {
                DRV_EVENT_MASK_uncore_capture(&event_flag) = 1;
            }
            if (ECB_entries_branch_evt_get(pecb, i)) {
                DRV_EVENT_MASK_branch(&event_flag) = 1;
            }
            if (ECB_entries_em_trigger_get(pecb,i)) {
                DRV_EVENT_MASK_trigger(&event_flag) = 1;
            }
            if (ECB_entries_collect_on_ctx_sw_get(pecb, i)) {
                DRV_EVENT_MASK_collect_on_ctx_sw(&event_flag) = 1;
            }

            if (DRV_MASKS_masks_num(masks) < MAX_OVERFLOW_EVENTS) {
                DRV_EVENT_MASK_bitFields1(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = DRV_EVENT_MASK_bitFields1(&event_flag);
                DRV_EVENT_MASK_event_idx(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_event_id_index(pecb, i);
                DRV_EVENT_MASK_desc_id(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_desc_id(pecb, i);
                DRV_MASKS_masks_num(masks)++;
            }
            else {
                SEP_DRV_LOG_TRACE("The array for event masks is full.");
            }
            SEP_DRV_LOG_TRACE("Overflow -- 0x%llx, index 0x%llx.", overflow_status, (U64)1 << index);
            SEP_DRV_LOG_TRACE("Slot# %d, reg_id 0x%x, index %d.",
                             i, ECB_entries_reg_id(pecb, i), index);
            if (ECB_entries_event_id_index(pecb, i) == CPU_STATE_trigger_event_num(pcpu)) {
                CPU_STATE_trigger_count(pcpu)--;
            }
        }
    } END_FOR_EACH_DATA_REG;

    CPU_STATE_reset_mask(pcpu) = overflow_status_clr;
    /* Clear outstanding overflow bits */
    SYS_Write_MSR(IA32_PERF_GLOBAL_OVF_CTRL, overflow_status_clr & PERFVER4_OVERFLOW_BIT_MASK_NON_HT);

    SEP_DRV_LOG_TRACE("Check Overflow completed %d.", this_cpu);
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn perfver4_Read_Counts(param, id)
 *
 * @param    param    The read thread node to process
 * @param    id       The event id for the which the sample is generated
 *
 * @return   None     No return needed
 *
 * @brief    Read CPU event based counts data and store into the buffer param;
 *           For the case of the trigger event, store the SAV value.
 */
static VOID
perfver4_Read_Counts (
    PVOID  param,
    U32    id
)
{
    U64            *data;
    U32             this_cpu            = CONTROL_THIS_CPU();
    CPU_STATE       pcpu                = &pcb[this_cpu];
    U32             dev_idx             = core_to_dev_map[this_cpu];
    DEV_CONFIG      pcfg                = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    U32             event_id            = 0;

    if (!DEV_CONFIG_num_events(pcfg)) {
        return;
    }

    if (DEV_CONFIG_ebc_group_id_offset(pcfg)) {
        // Write GroupID
        data  = (U64 *)((S8*)param + DEV_CONFIG_ebc_group_id_offset(pcfg));
        *data = CPU_STATE_current_group(pcpu) + 1;
    }

    LFENCE_SERIALIZE();
    FOR_EACH_DATA_REG(pecb,i) {
        if (ECB_entries_counter_event_offset(pecb,i) == 0) {
            continue;
        }
        data = (U64 *)((S8*)param + ECB_entries_counter_event_offset(pecb,i));
        event_id = ECB_entries_event_id_index(pecb,i);
        if (event_id == id) {
            *data = ~(ECB_entries_reg_value(pecb,i) - 1) &
                                           ECB_entries_max_bits(pecb,i);;
        }
        else {
            *data = SYS_Read_PMC(ECB_entries_reg_id(pecb,i), ECB_entries_fixed_reg_get(pecb, i));
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
        }
    } END_FOR_EACH_DATA_REG;
    LFENCE_SERIALIZE();

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn perfver4_Read_Metrics(buffer, id)
 *
 * @param    param        buffer to write metrics into
 *
 * @return   None     No return needed
 *
 * @brief    Read hardware metrics from IA32_PERF_METRICS MSR
 */
static VOID
perfver4_Read_Metrics (
    PVOID  buffer
)
{
    U64            *data, metrics = 0;
    U32             j;
    U32             index = 0;
    U32             this_cpu = CONTROL_THIS_CPU();
    U32             dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG      pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg)) {
        return;
    }

    data = (U64 *)buffer;

    FOR_EACH_NONEVENT_REG(pecb,i) {
        metrics = SYS_Read_MSR(ECB_entries_reg_id(pecb,i));
            for (j = 0; j < DEV_CONFIG_num_perf_metrics(pcfg); j++) {
            *data = (metrics & (0xFFULL << 8*j)) >> 8*j;
            data++;
        }
    } END_FOR_EACH_NONEVENT_REG;

    if (DRV_CONFIG_emon_mode(drv_cfg)) {
        SYS_Write_MSR(IA32_FIXED_CTR3, 0LL);
    }
    else {
        FOR_EACH_DATA_REG(pecb, i) {
            if (ECB_entries_fixed_reg_get(pecb, i)) {
                index = ECB_entries_reg_id(pecb, i) - IA32_FIXED_CTR0 + 0x20;
                if (index == FIXED_CTR3_BIT_INDEX) {
                    SYS_Write_MSR(IA32_FIXED_CTR3, ECB_entries_reg_value(pecb,i));
                    break;
                }
            }
        } END_FOR_EACH_DATA_REG;
    }
    SYS_Write_MSR(IA32_PERF_METRICS, 0LL);
    return;
}
/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 perfver4_Platform_Info
 *
 * @brief       Reads the MSR_PLATFORM_INFO register if present
 *
 * @param       void
 *
 * @return      value read from the register
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static VOID
perfver4_Platform_Info (
    PVOID data
)
{
    DRV_PLATFORM_INFO      platform_data = (DRV_PLATFORM_INFO)data;
    U64                    value         = 0;

    if (!platform_data) {
        return;
    }

#define IA32_MSR_PLATFORM_INFO 0xCE
    value = SYS_Read_MSR(IA32_MSR_PLATFORM_INFO);

    DRV_PLATFORM_INFO_info(platform_data)           = value;
    DRV_PLATFORM_INFO_ddr_freq_index(platform_data) = 0;
    SEP_DRV_LOG_TRACE ("perfver4_Platform_Info: Read from MSR_ENERGY_MULTIPLIER reg is %llu.", SYS_Read_MSR(MSR_ENERGY_MULTIPLIER));
    DRV_PLATFORM_INFO_energy_multiplier(platform_data) = (U32) (SYS_Read_MSR(MSR_ENERGY_MULTIPLIER) & 0x00001F00) >> 8;

    return;
}

/*
 * Initialize the dispatch table
 */
DISPATCH_NODE perfver4_dispatch =
{
    .init                     = perfver4_Initialize,
    .fini                     = perfver4_Destroy,
    .write                    = perfver4_Write_PMU,
    .freeze                   = perfver4_Disable_PMU,
    .restart                  = perfver4_Enable_PMU,
    .read_data                = perfver4_Read_PMU_Data,
    .check_overflow           = perfver4_Check_Overflow,
    .swap_group               = perfver4_Swap_Group,
    .read_lbrs                = perfver4_Read_LBRs,
    .clean_up                 = perfver4_Clean_Up,
    .hw_errata                = NULL,
    .read_power               = NULL,
    .check_overflow_errata    = NULL,
    .read_counts              = perfver4_Read_Counts,
    .check_overflow_gp_errata = NULL,
    .read_ro                  = NULL,
    .platform_info            = perfver4_Platform_Info,
    .trigger_read             = NULL,
    .scan_for_uncore          = NULL,
    .read_metrics             = NULL

};

DISPATCH_NODE perfver4_dispatch_htoff_mode =
{
    .init                     = perfver4_Initialize,
    .fini                     = perfver4_Destroy,
    .write                    = perfver4_Write_PMU,
    .freeze                   = perfver4_Disable_PMU,
    .restart                  = perfver4_Enable_PMU,
    .read_data                = perfver4_Read_PMU_Data,
    .check_overflow           = perfver4_Check_Overflow_Htoff_Mode,
    .swap_group               = perfver4_Swap_Group,
    .read_lbrs                = perfver4_Read_LBRs,
    .clean_up                 = perfver4_Clean_Up,
    .hw_errata                = NULL,
    .read_power               = NULL,
    .check_overflow_errata    = NULL,
    .read_counts              = perfver4_Read_Counts,
    .check_overflow_gp_errata = NULL,
    .read_ro                  = NULL,
    .platform_info            = perfver4_Platform_Info,
    .trigger_read             = NULL,
    .scan_for_uncore          = NULL,
    .read_metrics             = NULL
};

DISPATCH_NODE perfver4_dispatch_nonht_mode =
{
    .init                     = perfver4_Initialize,
    .fini                     = perfver4_Destroy,
    .write                    = perfver4_Write_PMU,
    .freeze                   = perfver4_Disable_PMU,
    .restart                  = perfver4_Enable_PMU,
    .read_data                = perfver4_Read_PMU_Data,
    .check_overflow           = perfver4_Check_Overflow_Nonht_Mode,
    .swap_group               = perfver4_Swap_Group,
    .read_lbrs                = perfver4_Read_LBRs,
    .clean_up                 = perfver4_Clean_Up,
    .hw_errata                = NULL,
    .read_power               = NULL,
    .check_overflow_errata    = NULL,
    .read_counts              = perfver4_Read_Counts,
    .check_overflow_gp_errata = NULL,
    .read_ro                  = NULL,
    .platform_info            = perfver4_Platform_Info,
    .trigger_read             = NULL,
    .scan_for_uncore          = NULL,
    .read_metrics             = perfver4_Read_Metrics
};
