# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

import os
from pathlib import Path

from mpp.core.validators import DirectoryValidator, ExcelFileValidator


class OutputHandler:
    EXCEL_EXTENSION = '.xlsx'
    EXCEL_FORMAT_STRING = 'xlsx'
    DEFAULT_EXCEL_FILE_NAME = 'summary.xlsx'
    DEFAULT_OUTPUT_FILE_PREFIX = '__mpp'

    class OutputHandlerOutputFormatError(ValueError):
        def __init__(self, output_format_string: str):
            super().__init__(f'File format {output_format_string} may only be specified once on the command line. '
                             f'For example, the combination '
                             f'\'-o <filename>.xlsx --output-format xlsx\' is invalid. '
                             f'Use \'-o <filename>.xlsx\' OR \'-o <file_prefix> --output-format xlsx\' '
                             f'instead.')

    def __init__(self, output_arg: str, output_format_arg: [None, str] = None):
        self.__output_directory: [None, str] = None
        self.__output_prefix: str = self.DEFAULT_OUTPUT_FILE_PREFIX
        self.__excel_file_name: [None, str] = None
        self.__set_output_directory(output_arg)
        self.__set_output_properties(output_arg, output_format_arg)

    @property
    def output_directory(self) -> str:
        return self.__output_directory

    @property
    def output_file_prefix(self) -> [None, str]:
        return self.__output_prefix

    @property
    def excel_file_name(self) -> [None, str]:
        return self.__excel_file_name

    @property
    def write_excel_file(self) -> bool:
        return True if self.__excel_file_name else False

    @property
    def excel_file_with_path(self) -> [None, str]:
        return os.path.join(self.output_directory, self.excel_file_name) if self.write_excel_file else None

    def __set_output_directory(self, output_arg):
        if os.path.isdir(output_arg):
            self.__output_directory = output_arg
        else:
            self.__output_directory = str(Path(output_arg).parent)

    def __set_output_properties(self, output_arg, output_format_arg: [None, str] = None):
        if self.__output_arg_is_excel_file(output_arg):
            self.__set_excel_filename_to_output_arg(output_arg, output_format_arg)
        elif self.__output_arg_is_file_prefix(output_arg):
            self.__output_prefix = os.path.basename(output_arg)
            self.__set_excel_filename_to_file_prefix_if_requested(output_format_arg)
        elif self.__output_arg_is_directory(output_arg):
            self.__set_excel_filename_default_if_requested(output_format_arg)
        self.__validate_output_files()

    def __output_arg_is_excel_file(self, output_arg: str) -> bool:
        return Path(output_arg).suffix == self.EXCEL_EXTENSION

    def __set_excel_filename_to_output_arg(self, output_arg: str, output_format_arg: [None, str] = None):
        if self.__output_format_excel(output_format_arg):
            raise self.OutputHandlerOutputFormatError(self.EXCEL_FORMAT_STRING)
        self.__excel_file_name = os.path.basename(output_arg)

    def __output_arg_is_file_prefix(self, output_arg: str) -> bool:
        return True if output_arg != self.output_directory else False

    def __set_excel_filename_to_file_prefix_if_requested(self, output_format_arg: [None, str] = None):
        if self.__output_format_excel(output_format_arg):
            self.__excel_file_name = self.__output_prefix + self.EXCEL_EXTENSION

    def __output_arg_is_directory(self, output_arg: str) -> bool:
        return True if output_arg == self.output_directory else False

    def __set_excel_filename_default_if_requested(self, output_format_arg: [None, str] = None):
        if self.__output_format_excel(output_format_arg):
            self.__excel_file_name = self.DEFAULT_EXCEL_FILE_NAME

    def __output_format_excel(self, output_format_arg: [None, str] = None):
        return True if output_format_arg and self.EXCEL_FORMAT_STRING in output_format_arg else False

    def __validate_output_files(self):
        directory_validator = DirectoryValidator()
        directory_validator(self.output_directory)
        if self.__excel_file_name:
            excel_file_validator = ExcelFileValidator()
            excel_file_validator(self.excel_file_name)
