# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

import sys
import logging
from cli import __version__


class ConsoleOutput:
    """
    A set of static methods to output information to the user console in main.py
    """

    @staticmethod
    def initialize_logging(verbose):
        logging_level = ConsoleOutput._get_logging_level(verbose)
        logging.basicConfig(level=logging_level, format='%(asctime)s %(levelname)s: %(message)s',
                    datefmt='%Y-%m-%d %I:%M:%S %p', stream=sys.stderr)

    @staticmethod
    def show_run_information(ref_tsc):
        print('System information:')
        print(f'     TSC Frequency: {ref_tsc // 1000000} MHz')
        print()
        print(f'Processing data: ...')

    @staticmethod
    def show_banner():
        print(f'Metric post processor version {__version__}')
        print(f'Python {sys.version}')

    @staticmethod
    def show_error(msg, exit_code=1, exception=None):
        """
        Print an error and terminate the script
        :param msg: error message to print
        :param exit_code: optional exit code (default is 1)
        :param exception: optional exception object
        """
        logging.exception(exception)
        if exception:
            msg += f': {exception}'
        print(msg, file=sys.stderr)
        exit(exit_code)

    @staticmethod
    def show_sample_count(first_sample, last_sample, num_unique_events, total_metrics_derived):
        sample_count = last_sample - first_sample + 1
        print(f'{sample_count} samples processed')
        print(f'{num_unique_events} events parsed and {len(total_metrics_derived)} metrics derived.')

    @staticmethod
    def _get_logging_level(verbose):
        logging_level = logging.WARNING
        if verbose:
            logging_level = logging.DEBUG
        return logging_level
