/*
 * Copyright (C) 2011 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/


#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_struct.h"

#include "lwpmudrv.h"
#include "utility.h"
#include "control.h"
#include "output.h"
#include "haswellunc_sa.h"
#include "ecb_iterators.h"
#include "pci.h"

static U64            sa_chap_virtual_address = 0;
static U32            chap_overflow[HSWUNC_SA_MAX_COUNTERS];

/*!
 * @fn          static VOID hswunc_sa_Write_PMU(VOID*)
 *
 * @brief       Initial write of PMU registers
 *              Walk through the entries and write the value of the register accordingly.
 *              When current_group = 0, then this is the first time this routine is called,
 *
 * @param       param - device index
 *
 * @return      None
 *
 * <I>Special Notes:</I>
 */
static VOID
hswunc_sa_Write_PMU (
    VOID  *param
)
{
    U32                        dev_idx  = *((U32*)param);
    int                        this_cpu = CONTROL_THIS_CPU();
    ECB                        pecb     = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[0];
    DRV_PCI_DEVICE_ENTRY       dpden;
    U64                        bar;
    U64                        physical_address;
    U32                        dev_index       = 0;
    S32                        bar_list[HSWUNC_SA_MAX_PCI_DEVICES];
    U32                        bar_index       = 0;
    U64                        gdxc_bar        = 0;
    U32                        map_size        = 0;
    U64                        virtual_address = 0;
    U64                        mmio_offset     = 0;
    U32                        bar_name        = 0;
    DRV_PCI_DEVICE_ENTRY       curr_pci_entry  = NULL;
    U32                        i               = 0;
    CPU_STATE                  pcpu            = &pcb[this_cpu];

    if (!CPU_STATE_system_master(pcpu)) {
        return;
    }

    if (!pecb) {
        return;
    }

    for (dev_index = 0; dev_index < HSWUNC_SA_MAX_PCI_DEVICES; dev_index++) {
        bar_list[dev_index] = -1;
    }

    // initialize the CHAP per-counter overflow numbers
    for (i = 0; i < HSWUNC_SA_MAX_COUNTERS; i++) {
        chap_overflow[i] = 0;
    }

    ECB_pcidev_entry_list(pecb) = (DRV_PCI_DEVICE_ENTRY)((S8*)pecb + ECB_pcidev_list_offset(pecb));
    dpden = ECB_pcidev_entry_list(pecb);

    for (dev_index = 0; dev_index < ECB_num_pci_devices(pecb); dev_index++) {
        curr_pci_entry = &dpden[dev_index];
        mmio_offset    = DRV_PCI_DEVICE_ENTRY_base_offset_for_mmio(curr_pci_entry);
        bar_name       = DRV_PCI_DEVICE_ENTRY_bar_name(curr_pci_entry);
        if (DRV_PCI_DEVICE_ENTRY_config_type(curr_pci_entry) == UNC_PCICFG) {
            PCI_Write_U32(0, DRV_PCI_DEVICE_ENTRY_bus_no(curr_pci_entry),
                        DRV_PCI_DEVICE_ENTRY_dev_no(curr_pci_entry),
                        DRV_PCI_DEVICE_ENTRY_func_no(curr_pci_entry),
                        mmio_offset,
                        DRV_PCI_DEVICE_ENTRY_value(curr_pci_entry));
            continue;
        }
        // UNC_MMIO programming
        if (bar_list[bar_name] != -1) {
            bar_index                                            = bar_list[bar_name];
            virtual_address                                      = DRV_PCI_DEVICE_ENTRY_virtual_address(&dpden[bar_index]);
            DRV_PCI_DEVICE_ENTRY_virtual_address(curr_pci_entry) = DRV_PCI_DEVICE_ENTRY_virtual_address(&dpden[bar_index]);
            writel((U32*)(((char*)(UIOP)virtual_address)+mmio_offset), DRV_PCI_DEVICE_ENTRY_value(curr_pci_entry));
            continue;
        }
        if (bar_name == UNC_GDXCBAR) {
            DRV_PCI_DEVICE_ENTRY_bar_address(curr_pci_entry) = gdxc_bar;
        }
        else {
            bar  = PCI_Read_U64(0, DRV_PCI_DEVICE_ENTRY_bus_no(curr_pci_entry),
                               DRV_PCI_DEVICE_ENTRY_dev_no(curr_pci_entry),
                               DRV_PCI_DEVICE_ENTRY_func_no(curr_pci_entry),
                               DRV_PCI_DEVICE_ENTRY_bar_offset(curr_pci_entry));

            bar &= HSWUNC_SA_BAR_ADDR_MASK;

            DRV_PCI_DEVICE_ENTRY_bar_address(curr_pci_entry) = bar;
        }
        physical_address = DRV_PCI_DEVICE_ENTRY_bar_address(curr_pci_entry);

        if (physical_address) {
            if (bar_name == UNC_MCHBAR) {
                map_size = HSWUNC_SA_MCHBAR_MMIO_PAGE_SIZE;
            }
            else if (bar_name == UNC_PCIEXBAR) {
                map_size = HSWUNC_SA_PCIEXBAR_MMIO_PAGE_SIZE;
            }
            else {
                map_size = HSWUNC_SA_OTHER_BAR_MMIO_PAGE_SIZE;
            }
            DRV_PCI_DEVICE_ENTRY_virtual_address(curr_pci_entry) = (U64) (UIOP)pmap_mapdev(physical_address, map_size);
            DRV_PCI_DEVICE_ENTRY_size(curr_pci_entry) = map_size;
            virtual_address = DRV_PCI_DEVICE_ENTRY_virtual_address(curr_pci_entry);

            if (!gdxc_bar && bar_name == UNC_MCHBAR) {
                gdxc_bar  = PCI_MMIO_Read_U64(virtual_address, HSWUNC_SA_GDXCBAR_OFFSET_LO);
                gdxc_bar &= HSWUNC_SA_GDXCBAR_MASK;
            }
            PCI_MMIO_Write_U32(virtual_address, mmio_offset, DRV_PCI_DEVICE_ENTRY_value(curr_pci_entry));
            bar_list[bar_name] = dev_index;
            if (sa_chap_virtual_address == 0 && bar_name == UNC_CHAPADR) {
                sa_chap_virtual_address = virtual_address;
            }
        }
    }

    return;
}

/*!
 * @fn         static VOID hswunc_sa_Disable_PMU(PVOID)
 *
 * @brief      Unmap the virtual address when sampling/driver stops
 *
 * @param      param - device index
 *
 * @return     None
 *
 * <I>Special Notes:</I>
 */
static VOID
hswunc_sa_Disable_PMU (
    PVOID  param
)
{

    U32                   this_cpu  = CONTROL_THIS_CPU();
    DRV_PCI_DEVICE_ENTRY  dpden;
    U32                   dev_index = 0;
    U32                   dev_idx   = *((U32*)param);
    ECB                   pecb      = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[0];
    U32                   i         = 0;
    CPU_STATE             pcpu      = &pcb[this_cpu];

    if (!CPU_STATE_system_master(pcpu)) {
        return;
    }

    if (!pecb) {
        return;
    }

    if (sa_chap_virtual_address) {
        for (i = 0; i < ECB_num_entries(pecb); i++) {
            writel((U32*)(((char*)(UIOP)sa_chap_virtual_address)+HSWUNC_SA_CHAP_CTRL_REG_OFFSET+i*0x10),
                   HSWUNC_SA_CHAP_STOP);
        }
    }

    dpden = ECB_pcidev_entry_list(pecb);
    for (dev_index = 0; dev_index < ECB_num_pci_devices(pecb); dev_index++) {
        if (DRV_PCI_DEVICE_ENTRY_config_type(&dpden[dev_index]) == UNC_MMIO &&
            DRV_PCI_DEVICE_ENTRY_bar_address(&dpden[dev_index]) != 0) {
#if __FreeBSD_version >= 1400000
            pmap_unmapdev((void *)DRV_PCI_DEVICE_ENTRY_virtual_address(&dpden[dev_index]),
                          DRV_PCI_DEVICE_ENTRY_size(&dpden[dev_index]));
#else
            pmap_unmapdev(DRV_PCI_DEVICE_ENTRY_virtual_address(&dpden[dev_index]),
                          DRV_PCI_DEVICE_ENTRY_size(&dpden[dev_index]));
#endif
        }
    }
    sa_chap_virtual_address = 0;

    return;
}

/*!
 * @fn         static VOID hswunc_sa_Initialize(PVOID)
 *
 * @brief      Initialize any registers or addresses
 *
 * @param      param
 *
 * @return     None
 *
 * <I>Special Notes:</I>
 */
static VOID
hswunc_sa_Initialize (
    VOID  *param
)
{
    sa_chap_virtual_address = 0;
    return;
}


/*!
 * @fn         static VOID hswunc_sa_Clean_Up(PVOID)
 *
 * @brief      Reset any registers or addresses
 *
 * @param      param
 *
 * @return     None
 *
 * <I>Special Notes:</I>
 */
static VOID
hswunc_sa_Clean_Up (
    VOID   *param
)
{
    sa_chap_virtual_address = 0;
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn hswunc_sa_Trigger_Read(param, id, read_from_intr)
 *
 * @param    param          Pointer to populate read data
 * @param    id             Device index
 * @param    read_from_intr Read data from interrupt or timer
 *
 * @return   None     No return needed
 *
 * @brief    Read the Uncore count data and store into the buffer param;
 *
 */
static VOID
hswunc_sa_Trigger_Read (
    PVOID  param,
    U32    id,
    U32    read_from_intr
)
{
    U64                  *data       = (U64*) param;
    ECB                   pecb       = LWPMU_DEVICE_PMU_register_data(&devices[id])[0];
    U32                   data_val     = 0;
    U64                   total_count  = 0;

    // group id
    data    = (U64*)((S8*)data + ECB_group_offset(pecb));
    *data   = 1;

    FOR_EACH_PCI_DATA_REG_RAW(pecb, i, id) {
        data  = (U64 *)((S8*)param + ECB_entries_counter_event_offset(pecb,i));
        writel((U32*)(((char*)(UIOP)sa_chap_virtual_address) + HSWUNC_SA_CHAP_CTRL_REG_OFFSET + i*0x10),
               HSWUNC_SA_CHAP_SAMPLE_DATA);
        data_val = readl((U32*)((char*)(UIOP)(sa_chap_virtual_address) + ECB_entries_reg_offset(pecb, i)));
        if (data_val < pcb[0].last_visa_count[i]) {
            chap_overflow[i]++;
        }
        pcb[0].last_visa_count[i] = data_val;
        total_count = data_val + chap_overflow[i]*HSWUNC_SA_MAX_COUNT;
        *data = total_count;
    } END_FOR_EACH_PCI_DATA_REG_RAW;

    return;
}




/*
 * Initialize the dispatch table
 */
DISPATCH_NODE  hswunc_sa_dispatch =
{
    hswunc_sa_Initialize,        // initialize
    NULL,                        // destroy
    hswunc_sa_Write_PMU ,        // write
    hswunc_sa_Disable_PMU,       // freeze
    NULL,                        // restart
    NULL,                        // read
    NULL,                        // check for overflow
    NULL,
    NULL,
    hswunc_sa_Clean_Up,
    NULL,
    NULL,
    NULL,
    NULL,                        //read_counts
    NULL,
    NULL,
    NULL,
    hswunc_sa_Trigger_Read,
    NULL,
    NULL
};
