/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ioctl.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_struct.h"

#include "lwpmudrv.h"
#include "control.h"
#include "output.h"
#include "utility.h"

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/condvar.h>
#include <sys/uio.h>

#define OTHER_C_DEVICES  1     // one for module

/*
 *  Global data: Buffer control structure
 *
 *  Note: flush_writers is the condition predicate, flush_mtx protects the
 *  wait queue and flush_cv provides the means to wait for a signal. It is
 *  not sufficient to use flush_cv as a gate, it will only work if the thread
 *  to be signaled is alteady waiting. Rather, the flush_writers predicate
 *  is used to determine if waiting is necessary, as soon as it reaches 0 the
 *  writers are done and there is no need to wait. The logic should be to
 *  take the lock, check the predicate, then wait if the predicate is not
 *  satisfied, repeating as needed.
 */
static atomic_t          flush_writers = 0;
static volatile int      flush         = 0;

static struct cv         flush_cv;
static struct mtx        flush_mtx;

extern DRV_BOOL          unc_buf_init;

/*
 *  @fn output_Free_Buffers(output, size)
 *
 *  @param    IN  outbuf      - The output buffer to manipulate
 *
 *  @brief   Deallocate the memory associated with the buffer descriptor
 *
 */
static VOID
output_Free_Buffers (
    BUFFER_DESC   buffer,
    size_t        size
)
{
    int       j;
    OUTPUT    outbuf;

    if (buffer == NULL) {
        return;
    }
    outbuf = &BUFFER_DESC_outbuf(buffer);
    for (j = 0; j < OUTPUT_NUM_BUFFERS; j++) {
        CONTROL_Free_Memory(OUTPUT_buffer(outbuf,j));
        OUTPUT_buffer(outbuf,j) = NULL;
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 *  @fn  int OUTPUT_Reserve_Buffer_Space (OUTPUT      outbuf,
 *                                        U32         size)
 *                                        boolean_t  *signal_full,
 *                                        U32         in_notification
 *
 *  @param  outbuf          IN output buffer to manipulate
 *  @param  size            IN The size of data to reserve
 *  @param  signal_full     IN Pointer to boolean to fill with buffer full status
 *  @param  in_notification IN Whether or not the function is called from a notification context
 *
 *  @result outloc - to the location where data is to be written
 *
 *  Reserve space in the output buffers for data.  If a buffer is full,
 *  signal the caller that the flush routine needs to be called.
 *
 * <I>Special Notes:</I>
 *
 */
extern void*
OUTPUT_Reserve_Buffer_Space (
    BUFFER_DESC  bd,
    U32          size,
    boolean_t   *signal_full,
    U32          in_notification
)
{
    char   *outloc      = NULL;
    OUTPUT  outbuf      = &BUFFER_DESC_outbuf(bd);

    SEP_DRV_LOG_NOTIFICATION_TRACE_IN(in_notification, "Bd: %p, size: %u, sig_full: %p, in_notif: %u.", bd, size, signal_full, in_notification);

    *signal_full = FALSE;
    if (OUTPUT_remaining_buffer_size(outbuf) >= size) {
        outloc = (OUTPUT_buffer(outbuf,OUTPUT_current_buffer(outbuf)) +
          (OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf)));
    }
    else {
        U32  i, j, start;
        OUTPUT_buffer_full(outbuf,OUTPUT_current_buffer(outbuf)) =
                OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf);

        //
        // Massive Naive assumption:  Must find a way to fix it.
        // In spite of the loop.
        // The next buffer to fill are monotonically increasing
        // indicies.
        //

        /*
         * Allow caller to determine how to handle case where a buffer is
         *  full, rather than calling cv_signal directly here.  This is
         *  because this routine may be called in NMI context, where
         *  we cannot do anything that will touch the scheduler (including
         *  cv_signal).
         */
        *signal_full = TRUE;
        start = OUTPUT_current_buffer(outbuf);
        for (i = start+1; i < start+OUTPUT_NUM_BUFFERS; i++) {

            j = i%OUTPUT_NUM_BUFFERS;

            if (!OUTPUT_buffer_full(outbuf,j)) {
                OUTPUT_current_buffer(outbuf) = j;
                OUTPUT_remaining_buffer_size(outbuf) = OUTPUT_BUFFER_SIZE;
                outloc = OUTPUT_buffer(outbuf,j);
            }
        }
    }
    if (outloc) {
        OUTPUT_remaining_buffer_size(outbuf) -= size;
        memset(outloc, 0, size);
    }

    SEP_DRV_LOG_NOTIFICATION_TRACE_OUT(in_notification, "Res: %p.", outloc);
    return outloc;
}

/* ------------------------------------------------------------------------- */
/*!
 *
 * @fn  int  OUTPUT_Buffer_Fill (BUFFER_DESC buf,
 *                               PVOID  data,
 *                               U16    size)
 *
 * @brief     Place a record (can be module, marker, etc) in a buffer
 *
 * @param     data - pointer to a buffer to copy
 * @param     size - size of the buffer to cpu
 *
 * @return    number of bytes copied into buffer
 *
 * Start by ensuring that output buffer space is available.
 * If so, then copy the input data to the output buffer and make the necessary
 * adjustments to manage the output buffers.
 * If not, signal the read event for this buffer and get another buffer.
 *
 * <I>Special Notes:</I>
 *
 */
static int
output_Buffer_Fill (
    BUFFER_DESC   bd,
    PVOID         data,
    U16           size,
    U32           in_notification
)
{
    char        *outloc;
    boolean_t   signal_full;

    SEP_DRV_LOG_NOTIFICATION_TRACE_IN(in_notification, "Bd: %p, Data: %p, size: %u, in_notif: %u.", bd, data, size, in_notification);

    outloc = OUTPUT_Reserve_Buffer_Space(bd, size, &signal_full, in_notification);
    if (signal_full) {
        mtx_lock(&BUFFER_DESC_mtx(bd));
        cv_signal(&BUFFER_DESC_cv(bd));
        mtx_unlock(&BUFFER_DESC_mtx(bd));
    }

    if (outloc) {
        memcpy(outloc, data, size);
        SEP_DRV_LOG_NOTIFICATION_TRACE_OUT(in_notification, "Res: %u.", size);
        return size;
    }

    SEP_DRV_LOG_NOTIFICATION_TRACE_OUT(in_notification, "Res: 0.");
    return 0;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  int  OUTPUT_Module_Fill (PVOID  data,
 *                               U16    size)
 *
 * @brief     Place a module record in a buffer
 *
 * @param     data - pointer to a buffer to copy
 * @param     size - size of the buffer to cpu
 *
 * @return    number of bytes copied into buffer
 *
 *
 */
extern int
OUTPUT_Module_Fill (
    PVOID     data,
    U16       size,
    U32       in_notification
)
{
    int     ret_size;
    OUTPUT  outbuf = &BUFFER_DESC_outbuf(module_buf);

    SEP_DRV_LOG_NOTIFICATION_TRACE_IN(in_notification, "Data: %p, size: %u, in_notif: %u.", data, size, in_notification);

    mtx_lock_spin(&OUTPUT_buffer_lock(outbuf));
    ret_size = output_Buffer_Fill(module_buf, data, size, in_notification);
    mtx_unlock_spin(&OUTPUT_buffer_lock(outbuf));

    SEP_DRV_LOG_NOTIFICATION_TRACE_OUT(in_notification, "Ret_size: %d.", ret_size);
    return ret_size;
}


/* ------------------------------------------------------------------------- */
/*!
 *  @fn  ssize_t  output_Read(struct file  *filp,
 *                            char         *buf,
 *                            size_t        count,
 *                            loff_t       *f_pos,
 *                            BUFFER_DESC   kernel_buf)
 *
 *  @brief  Return a sample buffer to user-mode. If not full or flush, wait
 *
 *  @param *filp          a file pointer
 *  @param *buf           a sampling buffer
 *  @param  count         size of the user's buffer
 *  @param  f_pos         file pointer (current offset in bytes)
 *  @param  kernel_buf    the kernel output buffer structure
 *
 *  @return number of bytes read. zero indicates end of file. Neg means error
 *
 *  Place no more than count bytes into the user's buffer.
 *  Block if unavailable on "BUFFER_DESC_queue(buf)"
 *
 * <I>Special Notes:</I>
 *
 */
static int
output_Read (
    struct cdev  *cdev,
    struct uio   *uio,
    int           flag,
    BUFFER_DESC   kernel_buf
)
{
    int      res    = OS_SUCCESS;
    ssize_t  to_copy;
    OUTPUT   outbuf = &BUFFER_DESC_outbuf(kernel_buf);
    U32      cur_buf, i;

/* Buffer is filled by output_fill_modules. */

    cur_buf = OUTPUT_current_buffer(outbuf);
    for (i=0; i<OUTPUT_NUM_BUFFERS; i++) { //iterate through all buffers
        cur_buf++;
        if (cur_buf >= OUTPUT_NUM_BUFFERS) { cur_buf = 0; } //circularly
        if ((to_copy = OUTPUT_buffer_full(outbuf, cur_buf))) {
            break;
        }
    }
    SEP_DRV_LOG_TRACE("buffer %d has %d bytes ready.", (S32)cur_buf, (S32)to_copy);
    if (!flush && to_copy == 0) {
        mtx_lock(&BUFFER_DESC_mtx(kernel_buf));
        SEP_DRV_LOG_TRACE("Read: before the loop!");
        while (cv_timedwait(&BUFFER_DESC_cv(kernel_buf), &BUFFER_DESC_mtx(kernel_buf), 1000) == EWOULDBLOCK && !flush) {
            SEP_DRV_LOG_TRACE("Read: in the loop!");
            if (DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_UNINITIALIZED | STATE_BIT_TERMINATING)) {
                res = ENXIO;
                break;
            }
        }
        SEP_DRV_LOG_TRACE("Read: after the loop!");
        mtx_unlock(&BUFFER_DESC_mtx(kernel_buf));

        SEP_DRV_LOG_TRACE("Get to copy.");
        to_copy = OUTPUT_buffer_full(outbuf, cur_buf);
        if (res == OS_SUCCESS) {
            SEP_DRV_LOG_TRACE("output_Read awakened, buffer %d has %d bytes.", cur_buf, (int)to_copy );
        }
        else {
            SEP_DRV_LOG_ERROR("output_Read has awakened due to abnormal driver state!");
            goto COPY_EXIT;
        }
    }

    if (DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_UNINITIALIZED | STATE_BIT_TERMINATING)) {
        SEP_DRV_LOG_ERROR("output_Read: Abnormal driver state, aborting.");
        res = ENXIO;
        goto COPY_EXIT;
    }

    /* Ensure that the user's buffer is large enough */
    if (!res && to_copy > uio->uio_resid) {
        SEP_DRV_LOG_ERROR("output_Read: User buffer is too small!");
        res = ENXIO;
        goto EXIT;
    }

    /* Copy data to user space. Note that we use cur_buf as the source */
COPY_EXIT:
    {
        int status;
        status = uiomove(OUTPUT_buffer(outbuf, cur_buf), to_copy, uio);
        /* Mark the buffer empty */
        OUTPUT_buffer_full(outbuf, cur_buf) = 0;
        if (status) {
            SEP_DRV_LOG_ERROR("output_Read: Failed to copy data to user space");
            res = ENXIO;
            goto EXIT;
        }
    }

    // At end-of-file, decrement the count of active buffer writers
EXIT:
    if (to_copy == 0){
        atomic_fetchadd_int(&flush_writers, -1);
        OUTPUT_buffer_full(outbuf, cur_buf) = 0;
        SEP_DRV_LOG_TRACE("output_Read decremented flush_writers.");
        if (atomic_load_acq_int(&flush_writers) == 0) {
            mtx_lock(&flush_mtx);
            cv_signal(&flush_cv);
            mtx_unlock(&flush_mtx);
        }
    }

    return res;
}

/* ------------------------------------------------------------------------- */
/*!
 *  @fn  ssize_t  OUTPUT_Module_Read(struct file  *filp,
 *                                   char         *buf,
 *                                   size_t        count,
 *                                   loff_t       *f_pos)
 *
 *  @brief  Return a module buffer to user-mode. If not full or flush, wait
 *
 *  @param *filp   a file pointer
 *  @param *buf    a sampling buffer
 *  @param  count  size of the user's buffer
 *  @param  f_pos  file pointer (current offset in bytes)
 *  @param  buf    the kernel output buffer structure
 *
 *  @return number of bytes read. zero indicates end of file. Neg means error
 *
 *  Place no more than count bytes into the user's buffer.
 *  Block on "BUFFER_DESC_queue(kernel_buf)" if buffer isn't full.
 *
 * <I>Special Notes:</I>
 *
 */
int
OUTPUT_Module_Read (
    struct cdev *cdev,
    struct uio  *uio,
    int          flag
)
{
    int res;
    SEP_DRV_LOG_TRACE("OUTPUT_Module_Read: in (module).");
    res = output_Read(cdev, uio, flag, module_buf);
    SEP_DRV_LOG_TRACE("OUTPUT_Module_Read: out (module): res = %d.", res);
    return res;
}


/* ------------------------------------------------------------------------- */
/*!
 *  @fn  ssize_t  OUTPUT_Sample_Read(struct file  *filp,
 *                                   char         *buf,
 *                                   size_t        count,
 *                                   loff_t       *f_pos)
 *
 *  @brief  Return a sample buffer to user-mode. If not full or flush, wait
 *
 *  @param *filp   a file pointer
 *  @param *buf    a sampling buffer
 *  @param  count  size of the user's buffer
 *  @param  f_pos  file pointer (current offset in bytes)
 *  @param  buf    the kernel output buffer structure
 *
 *  @return number of bytes read. zero indicates end of file. Neg means error
 *
 *  Place no more than count bytes into the user's buffer.
 *  Block on "BUFFER_DESC_queue(kernel_buf)" if buffer isn't full.
 *
 * <I>Special Notes:</I>
 *
 */
int
OUTPUT_Sample_Read (
    struct cdev *cdev,
    struct uio  *uio,
    int          flag
)
{
    int i;
    int res;

    i = (int)(long)cdev->si_drv1;

    SEP_DRV_LOG_TRACE("OUTPUT_Sample_Read: in (%d).", i);

    res = output_Read(cdev, uio, flag, &(cpu_buf[i]));

    SEP_DRV_LOG_TRACE("OUTPUT_Sample_Read: out (%d): res = %d.", i, res);
    return res;
}

/* ------------------------------------------------------------------------- */
/*!
 *  @fn  ssize_t  OUTPUT_Uncore_Read(struct file  *filp,
 *                                   char         *buf,
 *                                   size_t        count,
 *                                   loff_t       *f_pos)
 *
 *  @brief  Return a sample buffer to user-mode. If not full or flush, wait
 *
 *  @param *filp   a file pointer
 *  @param *buf    a sampling buffer
 *  @param  count  size of the user's buffer
 *  @param  f_pos  file pointer (current offset in bytes)
 *  @param  buf    the kernel output buffer structure
 *
 *  @return number of bytes read. zero indicates end of file. Neg means error
 *
 *  Place no more than count bytes into the user's buffer.
 *  Block on "BUFFER_DESC_queue(kernel_buf)" if buffer isn't full.
 *
 * <I>Special Notes:</I>
 *
 */
int
OUTPUT_Uncore_Read (
    struct cdev *cdev,
    struct uio  *uio,
    int          flag
)
{
    int i;
    int res;

    i = (int)(long)cdev->si_drv1;

    SEP_DRV_LOG_TRACE("OUTPUT_Uncore_Read: in (%d).", i);

    res = output_Read(cdev, uio, flag, &(unc_buf[i]));

    SEP_DRV_LOG_TRACE("OUTPUT_Sample_Read: out (%d): res = %d.", i, res);
    return res;
}

/* ------------------------------------------------------------------------- */
/*!
 *  @fn  ssize_t  OUTPUT_EMON_Read(struct file  *filp,
 *                                   char         *buf,
 *                                   size_t        count,
 *                                   loff_t       *f_pos)
 *
 *  @brief  Return EMON buffer to user-mode. If not full or flush, wait
 *
 *  @param *filp   a file pointer
 *  @param *buf    a sampling buffer
 *  @param  count  size of the user's buffer
 *  @param  f_pos  file pointer (current offset in bytes)
 *
 *  @return number of bytes read. zero indicates end of file. Neg means error
 *
 *  Place no more than count bytes into the user's buffer.
 *  Block on "BUFFER_DESC_queue(kernel_buf)" if buffer isn't full.
 *
 * <I>Special Notes:</I>
 *
 */
int
OUTPUT_Emon_Read (
    struct cdev *cdev,
    struct uio  *uio,
    int          flag
)
{
    int res;
    SEP_DRV_LOG_TRACE("OUTPUT_Emon_Read: in.");
    res = output_Read(cdev, uio, flag, emon_buf);
    SEP_DRV_LOG_TRACE("OUTPUT_Emon_Read: out: res = %d.", res);
    return res;
}

/*
 *  @fn output_Initialized_Buffers()
 *
 *  @result OUTPUT
 *  @brief  Allocate, initialize, and return an output data structure
 *
 * <I>Special Notes:</I>
 *     Multiple (OUTPUT_NUM_BUFFERS) buffers will be allocated
 *     Each buffer is of size (OUTPUT_BUFFER_SIZE)
 *     Each field in the buffer is initialized
 *     The event queue for the OUTPUT is initialized
 *
 */
static BUFFER_DESC
output_Initialized_Buffers (
    BUFFER_DESC desc
)
{
    OUTPUT       outbuf;
    int          j;

/*
 *  Allocate the BUFFER_DESC, then allocate its buffers
 */
    if (desc == NULL) {
        desc = (BUFFER_DESC)CONTROL_Allocate_Memory(sizeof(BUFFER_DESC_NODE));
        if (desc == NULL) {
            SEP_DRV_LOG_TRACE("OUTPUT Initialize_Buffer: Failed Allocation.");
            return(desc);
        }
    }

    if (!mtx_initialized(&OUTPUT_buffer_lock(&(BUFFER_DESC_outbuf(desc)))))
        mtx_init(&OUTPUT_buffer_lock(&(BUFFER_DESC_outbuf(desc))),
            "sep output buffer lock", NULL, MTX_SPIN);

    if (!mtx_initialized(&BUFFER_DESC_mtx(desc)))
        mtx_init(&BUFFER_DESC_mtx(desc), "buffer desc lock", NULL, MTX_DEF);


    outbuf = &(BUFFER_DESC_outbuf(desc));
    for (j = 0; j < OUTPUT_NUM_BUFFERS; j++) {
        if (OUTPUT_buffer(outbuf,j) == NULL) {
            OUTPUT_buffer(outbuf,j) = CONTROL_Allocate_Memory(OUTPUT_BUFFER_SIZE);
        }
        OUTPUT_buffer_full(outbuf,j) = 0;
        if (!OUTPUT_buffer(outbuf,j)) {
            SEP_DRV_LOG_TRACE("OUTPUT Initialize_Buffer: Failed Allocation.");
            return(desc);
        }
    }
    /*
     *  Initialize the remaining fields in the BUFFER_DESC
     */
    OUTPUT_current_buffer(outbuf)        = 0;
    OUTPUT_remaining_buffer_size(outbuf) = OUTPUT_BUFFER_SIZE;
    cv_init(&BUFFER_DESC_cv(desc), "sep buffer cv");

    return(desc);
}

/*
 *  @fn extern void OUTPUT_Initialize(void)
 *
 *  @param   buffer  -  seed name of the output file
 *  @param   len     -  length of the seed name
 *  @returns None
 *  @brief  Allocate, initialize, and return all output data structure
 *
 * <I>Special Notes:</I>
 *      Initialize the output structures.
 *      For each CPU in the system, allocate the output buffers.
 *      Initialize a module buffer and temp file to hold module information
 *      Initialize the read queues for each sample buffer
 *
 */
extern int
OUTPUT_Initialize (
    void
)
{
    BUFFER_DESC    unused;
    int            i;

    flush = 0;
    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        unused = output_Initialized_Buffers(&cpu_buf[i]);
        if (!unused) {
            SEP_DRV_LOG_ERROR("OUTPUT_Initialize: Failed to allocate cpu output buffers.");
            OUTPUT_Destroy();
            return OS_NO_MEM;
        }
    }

    /*
     *  Just need one module buffer
     */
    module_buf = output_Initialized_Buffers(module_buf);
    if (!module_buf) {
        SEP_DRV_LOG_ERROR("OUTPUT_Initialize: Failed to create module output buffers.");
        OUTPUT_Destroy();
        return OS_NO_MEM;
    }

    if (!mtx_initialized(&flush_mtx)) {
       cv_init(&flush_cv, "sep flush cv");
       mtx_init(&flush_mtx, "sep flush mtx", NULL, MTX_DEF);
    }

    return OS_SUCCESS;
}

/*
 *  @fn extern void OUTPUT_Initialize_UNC(void)
 *
 *  @param   buffer  -  seed name of the output file
 *  @param   len     -  length of the seed name
 *  @returns None
 *  @brief  Allocate, initialize, and return all output data structure
 *
 * <I>Special Notes:</I>
 *      Initialize the output structures.
 *      For each package in the system, allocate the output buffers.
 *
 */
extern int
OUTPUT_Initialize_UNC (
    void
)
{
    BUFFER_DESC    unused;
    int            i;

    for (i = 0; i < num_packages; i++) {
        unused = output_Initialized_Buffers(&unc_buf[i]);
        if (!unused) {
            SEP_DRV_LOG_ERROR("OUTPUT_Initialize: Failed to allocate uncore output buffers.");
            OUTPUT_Destroy();
            return OS_NO_MEM;
        }
    }

    return OS_SUCCESS;
}

/*
 *  @fn extern void OUTPUT_Initialize_EMON(void)
 *
 *  @returns OS_STATUS
 *  @brief  Allocate, initialize, and return EMON output data structure
 *
 * <I>Special Notes:</I>
 *      Initialize EMON buffer and temp file to hold EMON count data
 *
 */
extern int
OUTPUT_Initialize_EMON (
    void
)
{
    flush = 0;
    emon_buf = output_Initialized_Buffers(emon_buf);
    if (!emon_buf) {
        SEP_DRV_LOG_ERROR("OUTPUT_Initialize_EMON: Failed to create EMON output buffers.");
        OUTPUT_Destroy();
        return OS_NO_MEM;
    }

    return OS_SUCCESS;
}


/*
 *  @fn OS_STATUS  OUTPUT_Flush()
 *
 *  @brief  Flush the module buffers and sample buffers
 *
 *  @return OS_STATUS
 *
 *  For each CPU in the system, set buffer full to the byte count to flush.
 *  Flush the modules buffer, as well.
 *
 */
extern int
OUTPUT_Flush (
    VOID
)
{
    int        i;
    int        wait_count = 0;
    int        writers = 0;
    OUTPUT     outbuf;

    /*
     *  Flush all remaining data to files
     *  set up a flush event
     */
    SEP_DRV_LOG_TRACE_IN("Waiting for %d writers.",(GLOBAL_STATE_num_cpus(driver_state)+OTHER_C_DEVICES));

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        if (CPU_STATE_initial_mask(&pcb[i]) == 0) {
            continue;
        }
        outbuf = &(cpu_buf[i].outbuf);
        writers += 1;
        OUTPUT_buffer_full(outbuf,OUTPUT_current_buffer(outbuf)) =
            OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf);
    }
    if (unc_buf_init) {
        for (i = 0; i < num_packages; i++) {
            outbuf = &(unc_buf[i].outbuf);
            writers += 1;
            OUTPUT_buffer_full(outbuf,OUTPUT_current_buffer(outbuf)) =
                OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf);
        }
    }
    atomic_fetchadd_int(&flush_writers, writers + OTHER_C_DEVICES);
    // Flip the switch to terminate the output threads
    // Do not do this earlier, as threads may terminate before all the data is flushed
    flush = 1;
    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        if (CPU_STATE_initial_mask(&pcb[i]) == 0) {
            continue;
        }
        outbuf = &BUFFER_DESC_outbuf(&cpu_buf[i]);
        OUTPUT_buffer_full(outbuf,OUTPUT_current_buffer(outbuf)) =
            OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf);
        mtx_lock(&BUFFER_DESC_mtx(&cpu_buf[i]));
        cv_signal(&BUFFER_DESC_cv(&cpu_buf[i]));
        mtx_unlock(&BUFFER_DESC_mtx(&cpu_buf[i]));
    }

    if (unc_buf_init) {
        for (i = 0; i < num_packages; i++) {
            outbuf = &BUFFER_DESC_outbuf(&unc_buf[i]);
            OUTPUT_buffer_full(outbuf,OUTPUT_current_buffer(outbuf)) =
                OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf);
            mtx_lock(&BUFFER_DESC_mtx(&unc_buf[i]));
            cv_signal(&BUFFER_DESC_cv(&unc_buf[i]));
            mtx_unlock(&BUFFER_DESC_mtx(&unc_buf[i]));
        }
    }

    // Flush all data from the module buffers

    outbuf = &BUFFER_DESC_outbuf(module_buf);
    OUTPUT_buffer_full(outbuf,OUTPUT_current_buffer(outbuf)) =
                              OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf);
    SEP_DRV_LOG_TRACE("OUTPUT_Flush - waking up module_queue.");
    mtx_lock(&BUFFER_DESC_mtx(module_buf));
    cv_signal(&BUFFER_DESC_cv(module_buf));
    mtx_unlock(&BUFFER_DESC_mtx(module_buf));

    //Wait for buffers to empty
    SEP_DRV_LOG_TRACE("Before while sep flush wait loop...");
    mtx_lock(&flush_mtx);
    while (flush_writers) {
        if (cv_timedwait(&flush_cv, &flush_mtx, 1000) == EWOULDBLOCK) {
            if (DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_UNINITIALIZED | STATE_BIT_TERMINATING)) {
                SEP_DRV_LOG_ERROR("OUTPUT_Flush: Waiting for flush writers while in UNINITIALIZED or TERMINATING state! Aborting.");
                break;
            }
            if (wait_count == 1000) {
                SEP_DRV_LOG_ERROR("OUTPUT_Flush: TimeOut - Discarding current flush");
                atomic_fetchadd_int(&flush_writers, -(atomic_load_acq_int(&flush_writers)));
                break;
            }
            wait_count++;
        }
    }
    mtx_unlock(&flush_mtx);
    SEP_DRV_LOG_TRACE_OUT("Awakened from flush_queue.");
    flush = 0;

    return 0;
}

/*
 *  @fn OS_STATUS  OUTPUT_Flush_EMON()
 *
 *  @brief  Flush the EMON buffers
 *
 *  @return OS_STATUS
 *
 *  Set buffer full to the byte count to flush.
 *
 */
extern int
OUTPUT_Flush_EMON (
    VOID
)
{
    int        writers = 1;
    OUTPUT     outbuf;
    int        wait_count = 0;

    /*
     *  Flush all remaining data to files
     *  set up a flush event
     */
    SEP_DRV_LOG_TRACE_IN("Waiting for %d writers.",writers);
    atomic_fetchadd_int(&flush_writers, writers);

    // Flip the switch to terminate the output threads
    // Do not do this earlier, as threads may terminate before all the data is flushed
    flush = 1;

    // Flush all data from the EMON buffers
    outbuf = &BUFFER_DESC_outbuf(emon_buf);
    OUTPUT_buffer_full(outbuf,OUTPUT_current_buffer(outbuf)) =
                              OUTPUT_BUFFER_SIZE - OUTPUT_remaining_buffer_size(outbuf);
    SEP_DRV_LOG_TRACE("OUTPUT_Flush - waking up emon_queue.");
    mtx_lock(&BUFFER_DESC_mtx(emon_buf));
    cv_signal(&BUFFER_DESC_cv(emon_buf));
    mtx_unlock(&BUFFER_DESC_mtx(emon_buf));

    //Wait for buffers to empty
    SEP_DRV_LOG_TRACE("Before while EMON flush wait loop...");
    while (atomic_load_acq_int(&flush_writers) > 0) {
        DELAY(2000);
        if (DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_UNINITIALIZED | STATE_BIT_TERMINATING)) {
            SEP_DRV_LOG_ERROR("OUTPUT_Flush: Waiting for flush writers while in UNINITIALIZED or TERMINATING state! Aborting.");
            DELAY(1000000);
            break;
        }

        if (wait_count == 1000) {
            SEP_DRV_LOG_ERROR("OUTPUT_Flush: TimeOut - Discarding current flush");
            atomic_fetchadd_int(&flush_writers, -(atomic_load_acq_int(&flush_writers)));
            break;
        }
        wait_count++;
    }

    SEP_DRV_LOG_TRACE_OUT("Awakened from flush_queue.");
    flush = 0;

    return 0;
}

/*
 *  @fn extern void OUTPUT_Destroy()
 *
 *  @param   buffer  -  seed name of the output file
 *  @param   len     -  length of the seed name
 *  @returns OS_STATUS
 *  @brief   Deallocate output structures
 *
 * <I>Special Notes:</I>
 *      Free the module buffers
 *      For each CPU in the system, free the sampling buffers
 */
extern int
OUTPUT_Destroy (
    VOID
)
{
    int    i, n;
    output_Free_Buffers(module_buf, OUTPUT_BUFFER_SIZE);

    if (cpu_buf != NULL) {
        n = GLOBAL_STATE_num_cpus(driver_state);
        for (i = 0; i < n; i++) {
            output_Free_Buffers(&cpu_buf[i], OUTPUT_BUFFER_SIZE);
        }
    }

    if (unc_buf != NULL) {
        n = num_packages;
        for (i = 0; i < n; i++) {
            output_Free_Buffers(&unc_buf[i], OUTPUT_BUFFER_SIZE);
        }
    }

    if (emon_buf) {
        output_Free_Buffers(emon_buf, OUTPUT_BUFFER_SIZE);
    }

    flush_writers = 0;

    return 0;
}
