/*******************************************************************************
* Copyright 2024 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP API, oneapi::mkl::sparse::omatconvert, to perform an out-of-place
*       conversion operation between two sparse matrices on a SYCL device (CPU,
*       GPU). A conversion from a matrix in CSR format to a matrix in COO
*       format is presented.
*
*           A(CSR) -> B(COO)
*
*       The supported floating point data types for omatconvert matrix data are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported conversions for omatconvert are:
*           COO -> CSR
*           CSR -> COO
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix conversion: A(COO) -> B(CSR)
//
template <typename fpType, typename intType>
int run_sparse_matrix_omatconvert_example(const sycl::device &dev)
{
    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::gemv:\n"
                          << e.what() << std::endl;
            }
        }
    };

    // Create execution main_queue and buffers of matrix data
    sycl::queue main_queue(dev, exception_handler);

    // Initialize data for Sparse Matrix
    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::one;
    oneapi::mkl::index_base b_index = oneapi::mkl::index_base::zero;
    intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    intType b_ind = b_index == oneapi::mkl::index_base::zero ? 0 : 1;

    //
    // Set dimensions of matrices
    //
    intType size = 4;

    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;
    intType b_nrows = a_nrows;
    intType b_ncols = a_ncols;
    intType b_nnz = 0; // potentially unknown for now

    //
    // Set A data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fpType, mkl_allocator<fpType, 64>> a;

    ia.resize(a_nrows + 1);
    ja.resize(27 * a_nrows);
    a.resize(27 * a_nrows);

    generate_sparse_matrix<fpType, intType>(size, ia, ja, a, a_ind);
    a_nnz = ia[a_nrows] - a_ind;

    //
    // Declare B arrays
    //
    std::vector<intType, mkl_allocator<intType, 64>> ib;
    std::vector<intType, mkl_allocator<intType, 64>> jb;
    std::vector<fpType, mkl_allocator<fpType, 64>> b;

    //
    // omatconvert descriptor for conversion to COO
    //
    oneapi::mkl::sparse::omatconvert_descr_t descr = nullptr;

    //
    // Execute Matrix Conversion
    //

    std::cout << "\n\t\tsparse::omatconvert parameters:\n";

    std::cout << "\t\t\tA format = CSR" << std::endl;
    std::cout << "\t\t\tB format = COO" << std::endl;

    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;

    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tB_index = " << b_index << std::endl;

    oneapi::mkl::sparse::matrix_handle_t A = nullptr;
    oneapi::mkl::sparse::matrix_handle_t B = nullptr;

    sycl::buffer<intType, 1> a_rowptr(ia.data(), a_nrows + 1);
    sycl::buffer<intType, 1> a_colind(ja.data(), a_nnz);
    sycl::buffer<fpType, 1> a_vals(a.data(), a_nnz);

    std::int64_t size_temp_workspace;
    oneapi::mkl::sparse::omatconvert_alg alg =
        oneapi::mkl::sparse::omatconvert_alg::default_alg;

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);
        oneapi::mkl::sparse::init_matrix_handle(&B);

        init_omatconvert_descr(main_queue, &descr);

        // Set A matrix
        oneapi::mkl::sparse::set_csr_data(main_queue, A, a_nrows, a_ncols,
                                          a_index, a_rowptr, a_colind, a_vals);

        // Set B matrix with dummy data for now
        sycl::buffer<intType, 1> b_rowind_dummy(0);
        sycl::buffer<intType, 1> b_colind_dummy(0);
        sycl::buffer<fpType, 1> b_vals_dummy(0);

        oneapi::mkl::sparse::set_coo_data(main_queue, B, b_nrows, b_ncols,
                                          0, b_index, b_rowind_dummy,
                                          b_colind_dummy, b_vals_dummy);

        // Get the size of the temporary workspace for the subsequent stages
        oneapi::mkl::sparse::omatconvert_buffer_size(
            main_queue, A, B, alg, descr, size_temp_workspace);
        sycl::buffer<std::uint8_t, 1> temp_workspace(0, size_temp_workspace);

        // Analyze stage
        oneapi::mkl::sparse::omatconvert_analyze(
            main_queue, A, B, alg, descr, &temp_workspace);

        // Get NNZ for the B matrix
        std::int64_t b_nnz_out = 0;
        oneapi::mkl::sparse::omatconvert_get_nnz(
            main_queue, A, B, alg, descr, b_nnz_out);
        b_nnz = static_cast<intType>(b_nnz_out);

        // Set B matrix with properly allocated data
        sycl::buffer<intType, 1> b_rowind(0, b_nnz);
        sycl::buffer<intType, 1> b_colind(0, b_nnz);
        sycl::buffer<fpType, 1> b_vals(0, b_nnz);

        oneapi::mkl::sparse::set_coo_data(main_queue, B, b_nrows, b_ncols,
                                          b_nnz, b_index, b_rowind, b_colind,
                                          b_vals);

        // Convert A(CSR) -> B(COO)
        oneapi::mkl::sparse::omatconvert(main_queue, A, B, alg, descr);

        // Print portion of B solution
        {
            auto ib = b_rowind.get_host_access(sycl::read_only);
            auto jb = b_colind.get_host_access(sycl::read_only);
            auto b  = b_vals.get_host_access(sycl::read_only);
            std::cout << "B (COO) matrix [first ten elements]:" << std::endl;
            intType printed_values = std::min(static_cast<intType>(10),
                                              static_cast<intType>(b_nnz));
            for (intType i = 0; i < printed_values; ++i) {
                intType row = ib[i];
                intType col = jb[i];
                fpType val  = b[i];
                std::cout << "B(" << row << ", " << col << ") = " << val
                    << std::endl;
            }
        }

        // Clean up
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B);
        release_omatconvert_descr(main_queue, descr);
    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what()
            << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B);
        release_omatconvert_descr(main_queue, descr);

        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B);
        release_omatconvert_descr(main_queue, descr);

        return 1;
    }

    main_queue.wait();

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "#############################################################"
                 "###########"
              << std::endl;
    std::cout << "# Sparse Matrix Conversion Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#    A(CSR) -> B(COO)" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A and B are sparse matrices in CSR and COO formats"
                 " respectively" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::omatconvert_buffer_size" << std::endl;
    std::cout << "#   sparse::omatconvert_analyze" << std::endl;
    std::cout << "#   sparse::omatconvert_get_nnz" << std::endl;
    std::cout << "#   sparse::omatconvert" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::set_coo_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "#   sparse::init_omatconvert_descr" << std::endl;
    std::cout << "#   sparse::release_omatconvert_descr" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#############################################################"
                 "###########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_omatconvert_example<float, std::int32_t>(
                my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_omatconvert_example<double, std::int32_t>(my_dev);
                if (status != 0)
                    return status;
            }

            std::cout << "\tRunning with single precision complex data type:" << std::endl;
            status = run_sparse_matrix_omatconvert_example<std::complex<float>, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision complex data type:" << std::endl;
                status = run_sparse_matrix_omatconvert_example<std::complex<double>, std::int32_t>(my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    return status;
}
