=pod

=head1 COPYRIGHT

# (c) 1992-2024 Intel Corporation.                                              
# Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack         
# words and logos are trademarks of Intel Corporation or its                    
# subsidiaries in the U.S. and/or other countries. Other marks and              
# brands may be claimed as the property of others.                              
# See Trademarks on intel.com for full list of Intel trademarks or the          
# Trademarks & Brands Names Database (if Intel)                                 
# or See www.Intel.com/legal (if Altera)                                        
# Your use of Intel Corporation's design tools, logic functions and             
# other software and tools, and its AMPP partner logic functions, and           
# any output files any of the foregoing (including device programming           
# or simulation files), and any associated documentation or information         
# are expressly subject to the terms and conditions of the Altera               
# Program License Subscription Agreement, Intel MegaCore Function               
# License Agreement, or other applicable license agreement, including,          
# without limitation, that your use is for the sole purpose of                  
# programming logic devices manufactured by Intel and sold by Intel or          
# its authorized distributors.                                                  
# Please refer to the applicable agreement for further details.                 


=cut

package acl::Report_reader;
require acl::Common;
require JSON::PP;

our @ISA    = qw(Exporter);
our @EXPORT = qw(
  get_family_device
  get_quartus_version
  get_command
  get_version
  parse_info
  get_name
);
our @EXPORT_OK = qw(
  parse_details
  try_to_parse
);
our $VERSION = '1';

use strict;

# Returns the name of the object, i.e. loop, dependency variable
sub get_name ($) {
  my $my_obj = shift;
  return $my_obj->{"name"};
}

# Generic ACL JSON to string function
# return 0 when can't open the file instead of die because this parsing is used everywhere in the driver
sub try_to_parse($) {
  my $json_file = shift;
  open(my $json_fh, "<${json_file}") or return 0;
  my $json_txt = do {
    local $/;
    <$json_fh>
  };
  close $json_fh;
  
  my $json_to_perl;
  eval {
    $json_to_perl = JSON::PP::decode_json($json_txt);
  };
  if ($@ or !($json_to_perl)) {
    # deal with bad JSON format
    return undef;
  }
  my %ret = undef;
  eval {
    %ret = %{$json_to_perl};
  };
  if ($@ or !(%ret)) {
    return undef;
  } else {
    return %ret;
  }
}

# ==== Info specific  ==== #

sub get_family_device($) {
  my $my_obj = shift;
  return $my_obj->{"family"};
}

sub get_quartus_version($) {
  my $my_obj = shift;
  return $my_obj->{"quartus"};
}

sub get_command($) {
  my $my_obj = shift;
  return $my_obj->{"command"} if (defined($my_obj->{"command"}));
  return 0;
}

sub get_version($) {
  my $my_obj = shift;
  return $my_obj->{"version"};
}

=head2 parse_info($info_json_file)

Returns the info object if exist and the API above can be used to get the individual information.

=cut

sub parse_info($) {
  my $info_json_file = shift;
  my %info_dict = try_to_parse($info_json_file);
  if (defined($info_dict{"compileInfo"}) && defined($info_dict{"compileInfo"}->{"nodes"})) {
    my @info_list = @{$info_dict{"compileInfo"}->{"nodes"}};
    foreach my $info_obj (@info_list) {
      return $info_obj;
    }
  }
  return 0;
}


1;

__END__

=head1 NAME

acl::Report_reader - Report reader utilities

=head1 VERSION

$Header$

=head1 SYNOPSIS

=head1 DESCRIPTION

This module provides utilities for anyone who wishes to parse the report json files.

All methods names may optionally be imported, e.g.:

   use acl::Report_reader qw( parse_info );

=back
