/*******************************************************************************
* Copyright 2016 Intel Corporation.
*
*
* This software and the related documents are Intel copyrighted materials, and your use of them is governed by
* the express license under which they were provided to you ('License'). Unless the License provides otherwise,
* you may not use, modify, copy, publish, distribute, disclose or transmit this software or the related
* documents without Intel's prior written permission.
* This software and the related documents are provided as is, with no express or implied warranties, other than
* those that are expressly stated in the License.
*******************************************************************************/

#include "iw/iw_image_op.h"
#include "iw_owni.h"

IW_DECL(IppStatus) llwiAddC(const double *pValue, const void *pSrc, IwSize srcStep, void *pDst, IwSize dstStep,
                              IwiSize size, IppDataType dataType, int channels, int scaleFactor, IwiChDescriptor chDesc);

/**/////////////////////////////////////////////////////////////////////////////
//                   iwiAddC
///////////////////////////////////////////////////////////////////////////// */
IW_DECL(IppStatus) iwiAddC(const double *pValue, int valuesNum, const IwiImage *pSrcImage, IwiImage *pDstImage, const IwiAddCParams *pAuxParams, const IwiTile *pTile)
{
    IppStatus     status;
    IwiAddCParams auxParams;

    if(!pValue)
        return ippStsNullPtrErr;
    status = owniCheckImageRead(pSrcImage);
    if(status)
        return status;
    status = owniCheckImageWrite(pDstImage);
    if(status)
        return status;

    if(pSrcImage->m_typeSize != pDstImage->m_typeSize ||
        pSrcImage->m_channels != pDstImage->m_channels)
        return ippStsBadArgErr;

    if(pAuxParams)
        auxParams = *pAuxParams;
    else
        iwiAddC_SetDefaultParams(&auxParams);

    {
        const void *pSrc = pSrcImage->m_ptrConst;
        void       *pDst = pDstImage->m_ptr;
        IwiSize     size = owniGetMinSize(&pSrcImage->m_size, &pDstImage->m_size);

        if(pTile && pTile->m_initialized != ownTileInitNone)
        {
            if(pTile->m_initialized == ownTileInitSimple)
            {
                IwiRoi dstRoi = pTile->m_dstRoi;
                if(!owniTile_BoundToSize(&dstRoi, &size))
                    return ippStsNoOperation;

                pSrc = iwiImage_GetPtrConst(pSrcImage, dstRoi.y, dstRoi.x, 0);
                pDst = iwiImage_GetPtr(pDstImage, dstRoi.y, dstRoi.x, 0);
            }
            else if(pTile->m_initialized == ownTileInitPipe)
            {
                IwiRoi srcLim;
                IwiRoi dstLim;
                iwiTilePipeline_GetBoundedSrcRoi(pTile, &srcLim);
                iwiTilePipeline_GetBoundedDstRoi(pTile, &dstLim);

                pSrc = iwiImage_GetPtrConst(pSrcImage, srcLim.y, srcLim.x, 0);
                pDst = iwiImage_GetPtr(pDstImage, dstLim.y, dstLim.x, 0);

                size = owniGetMinSizeFromRect(&srcLim, &dstLim);
            }
            else
                return ippStsContextMatchErr;
        }

        if(valuesNum < pDstImage->m_channels)
        {
            int    i;
            Ipp64f val[4];
            for(i = 0; i < valuesNum; i++)
                val[i] = pValue[i];
            for(i = valuesNum; i < pDstImage->m_channels; i++)
                val[i] = pValue[valuesNum-1];

            return llwiAddC(val, pSrc, pSrcImage->m_step, pDst, pDstImage->m_step, size, pSrcImage->m_dataType, pSrcImage->m_channels, auxParams.scaleFactor, auxParams.chDesc);
        }

        return llwiAddC(pValue, pSrc, pSrcImage->m_step, pDst, pDstImage->m_step, size, pSrcImage->m_dataType, pSrcImage->m_channels, auxParams.scaleFactor, auxParams.chDesc);
    }
}


/**/////////////////////////////////////////////////////////////////////////////
//                   Low-Level Wrappers
///////////////////////////////////////////////////////////////////////////// */
IW_DECL(IppStatus) llwiAddC(const double *pValue, const void *pSrc, IwSize srcStep, void *pDst, IwSize dstStep,
                              IwiSize size, IppDataType dataType, int channels, int scaleFactor, IwiChDescriptor chDesc)
{
    OwniChCodes chCode = owniChDescriptorToCode(chDesc, channels, channels);
    Ipp64f      value[4];

    if(pSrc == pDst)
    {
        switch(dataType)
        {
#if IW_ENABLE_DATA_TYPE_8U
        case ipp8u:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_8u_C1IRSfs_L(ownCast_64f8u(*pValue), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_8u_C3IRSfs_L(ownCastArray_64f8u(pValue, value, 3), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_8u_C4IRSfs_L(ownCastArray_64f8u(pValue, value, 4), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_8u_AC4IRSfs_L(ownCastArray_64f8u(pValue, value, 4), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
#if IW_ENABLE_DATA_TYPE_16U
        case ipp16u:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_16u_C1IRSfs_L(ownCast_64f16u(*pValue), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_16u_C3IRSfs_L(ownCastArray_64f16u(pValue, value, 3), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_16u_C4IRSfs_L(ownCastArray_64f16u(pValue, value, 4), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_16u_AC4IRSfs_L(ownCastArray_64f16u(pValue, value, 4), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
#if IW_ENABLE_DATA_TYPE_16S
        case ipp16s:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_16s_C1IRSfs_L(ownCast_64f16s(*pValue), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_16s_C3IRSfs_L(ownCastArray_64f16s(pValue, value, 3), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_16s_C4IRSfs_L(ownCastArray_64f16s(pValue, value, 4), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_16s_AC4IRSfs_L(ownCastArray_64f16s(pValue, value, 4), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
#if IW_ENABLE_DATA_TYPE_32F
        case ipp32f:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_32f_C1IR_L(ownCast_64f32f(*pValue), (Ipp32f*)pDst, dstStep, size);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_32f_C3IR_L(ownCastArray_64f32f(pValue, value, 3), (Ipp32f*)pDst, dstStep, size);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_32f_C4IR_L(ownCastArray_64f32f(pValue, value, 4), (Ipp32f*)pDst, dstStep, size);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_32f_AC4IR_L(ownCastArray_64f32f(pValue, value, 4), (Ipp32f*)pDst, dstStep, size);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
        default: return ippStsDataTypeErr;
        }
    }
    else
    {
        switch(dataType)
        {
#if IW_ENABLE_DATA_TYPE_8U
        case ipp8u:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_8u_C1RSfs_L((Ipp8u*)pSrc, srcStep, ownCast_64f8u(*pValue), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_8u_C3RSfs_L((Ipp8u*)pSrc, srcStep, ownCastArray_64f8u(pValue, value, 3), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_8u_C4RSfs_L((Ipp8u*)pSrc, srcStep, ownCastArray_64f8u(pValue, value, 4), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_8u_AC4RSfs_L((Ipp8u*)pSrc, srcStep, ownCastArray_64f8u(pValue, value, 4), (Ipp8u*)pDst, dstStep, size, scaleFactor);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
#if IW_ENABLE_DATA_TYPE_16U
        case ipp16u:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_16u_C1RSfs_L((Ipp16u*)pSrc, srcStep, ownCast_64f16u(*pValue), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_16u_C3RSfs_L((Ipp16u*)pSrc, srcStep, ownCastArray_64f16u(pValue, value, 3), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_16u_C4RSfs_L((Ipp16u*)pSrc, srcStep, ownCastArray_64f16u(pValue, value, 4), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_16u_AC4RSfs_L((Ipp16u*)pSrc, srcStep, ownCastArray_64f16u(pValue, value, 4), (Ipp16u*)pDst, dstStep, size, scaleFactor);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
#if IW_ENABLE_DATA_TYPE_16S
        case ipp16s:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_16s_C1RSfs_L((Ipp16s*)pSrc, srcStep, ownCast_64f16s(*pValue), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_16s_C3RSfs_L((Ipp16s*)pSrc, srcStep, ownCastArray_64f16s(pValue, value, 3), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_16s_C4RSfs_L((Ipp16s*)pSrc, srcStep, ownCastArray_64f16s(pValue, value, 4), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_16s_AC4RSfs_L((Ipp16s*)pSrc, srcStep, ownCastArray_64f16s(pValue, value, 4), (Ipp16s*)pDst, dstStep, size, scaleFactor);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
#if IW_ENABLE_DATA_TYPE_32F
        case ipp32f:
            switch(chCode)
            {
#if IW_ENABLE_CHANNELS_C1
            case owniC1:        return ippiAddC_32f_C1R_L((Ipp32f*)pSrc, srcStep, ownCast_64f32f(*pValue), (Ipp32f*)pDst, dstStep, size);
#endif
#if IW_ENABLE_CHANNELS_C3
            case owniC3:        return ippiAddC_32f_C3R_L((Ipp32f*)pSrc, srcStep, ownCastArray_64f32f(pValue, value, 3), (Ipp32f*)pDst, dstStep, size);
#endif
#if IW_ENABLE_CHANNELS_C4
            case owniC4:        return ippiAddC_32f_C4R_L((Ipp32f*)pSrc, srcStep, ownCastArray_64f32f(pValue, value, 4), (Ipp32f*)pDst, dstStep, size);
#endif
#if IW_ENABLE_CHANNELS_AC4
            case owniC4M1110:   return ippiAddC_32f_AC4R_L((Ipp32f*)pSrc, srcStep, ownCastArray_64f32f(pValue, value, 4), (Ipp32f*)pDst, dstStep, size);
#endif
            default:            return ippStsNumChannelsErr;
            }
#endif
        default: return ippStsDataTypeErr;
        }
    }
}
