/*******************************************************************************
* Copyright 2021-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
!  Content:
!    Calculation of basic statistics Example Program Text
!******************************************************************************/

#include <stdio.h>
#include <math.h>

#include "mkl.h"
#include "mkl_omp_offload.h"

#include "errcheck.inc"

#define DIM     7        /* Task dimension */
#define N       1883     /* Number of observations */

#define P_THRESHOLD     0.01f

#define SEED    1
#define BRNG    VSL_BRNG_PHILOX4X32X10

int main(void)
{
    int dnum = 0;
    VSLSSTaskPtr task;
    MKL_INT dim;
    MKL_INT n;
    MKL_INT x_storage;
    float* x = (float*)mkl_malloc((N * DIM) * sizeof(float), 64);
    float* min_estimate = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* max_estimate = (float*)mkl_malloc(DIM * sizeof(float), 64);

    float* mean = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* raw2 = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* raw3 = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* raw4 = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* cen2 = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* cen3 = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* cen4 = (float*)mkl_malloc(DIM * sizeof(float), 64);

    float* variation = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* skewness = (float*)mkl_malloc(DIM * sizeof(float), 64);
    float* kurtosis = (float*)mkl_malloc(DIM * sizeof(float), 64);

    int i, j;
    int errcode;
    unsigned MKL_INT64 estimate = 0;
    int errnums = 0;

    /***** Initializing parameters for Summary Statistics task *****/
    dim         = DIM;
    n           = N;
    x_storage   = VSL_SS_MATRIX_STORAGE_ROWS;

    /***** Generate dataset *****/
    VSLStreamStatePtr stream;
    errcode = vslNewStream(&stream, BRNG, SEED);
    CheckVslError(errcode);
    errcode = vsRngUniform(0, stream, N * DIM, x, 0.0f, 0.1f);
    CheckVslError(errcode);
    errcode = vslDeleteStream(&stream);
    CheckVslError(errcode);

    for(i = 0; i < dim; i++)
    {
        min_estimate[i] = x[i*N];
        max_estimate[i] = x[i*N];
    }

#pragma omp target data map(tofrom:x[0:(N * DIM)], min_estimate[0:DIM], max_estimate[0:DIM], \
                            mean[0:DIM], raw2[0:DIM], raw3[0:DIM], raw4[0:DIM], \
                            cen2[0:DIM], cen3[0:DIM], cen4[0:DIM], variation[0:DIM], skewness[0:DIM], kurtosis[0:DIM]) device(dnum)
    {
        #pragma omp target data use_device_ptr(x, min_estimate, max_estimate, \
                mean, raw2, raw3, raw4, cen2, cen3, cen4, variation, skewness, kurtosis)
        {
            /***** Create Summary Statistics task *****/
            errcode = vslsSSNewTask( &task, &dim, &n, &x_storage, x, 0, 0 );
            CheckVslError(errcode);

            /***** Edit task parameters for min and max computation *****/
            errcode = vslsSSEditTask( task, VSL_SS_ED_MIN, min_estimate );
            CheckVslError(errcode);

            errcode = vslsSSEditTask( task, VSL_SS_ED_MAX, max_estimate );
            CheckVslError(errcode);

            /***** Edit task parameters for computatin of mean estimate and 2nd, 3rd
                   and 4th raw and central moments estimates *****/
            errcode = vslsSSEditMoments( task, mean, raw2, raw3, raw4, cen2, cen3, cen4 );
            CheckVslError(errcode);

            /***** Edit task parameters for kurtosis, skewness and variation
                   computation *****/
            errcode = vslsSSEditTask( task, VSL_SS_ED_KURTOSIS, kurtosis );
            CheckVslError(errcode);

            errcode = vslsSSEditTask( task, VSL_SS_ED_SKEWNESS, skewness );
            CheckVslError(errcode);

            errcode = vslsSSEditTask( task, VSL_SS_ED_VARIATION, variation );
            CheckVslError(errcode);

            /***** Minimum and maximum are included in the list of estimates
                to compute *****/
            estimate |= VSL_SS_MIN | VSL_SS_MAX;

            /***** Mean and 2nd, 3rd and 4th raw and central moments are included
                   in the list of estimates to compute *****/
            estimate |= VSL_SS_MEAN |
                VSL_SS_2R_MOM | VSL_SS_3R_MOM | VSL_SS_4R_MOM |
                VSL_SS_2C_MOM | VSL_SS_3C_MOM |
                VSL_SS_4C_MOM;

            /***** Kurtosis, skewness and variation are included in the list
                   of estimates to compute *****/
            estimate |= VSL_SS_KURTOSIS | VSL_SS_SKEWNESS | VSL_SS_VARIATION;

            /***** Compute the estimates using FAST method *****/
            #pragma omp dispatch device(dnum)
                errcode = vslsSSCompute( task, estimate, VSL_SS_METHOD_FAST );
        }
    }
    CheckVslError(errcode);

    /***** Testing stat characteristics of the computed estimates *****/

    /* Comparison of observations with min and max estimates */
    for(i = 0; i < dim; i++)
    {
        for(j = 0; j < n; j++)
        {
            if(x[j + i * N] < min_estimate[i]) {
                errnums++;
            }
            if(x[j + i * N] > max_estimate[i]) {
                errnums++;
            }
        }
    }

    /* Check moments */
    float mean_th[DIM], raw2_th[DIM], raw3_th[DIM], raw4_th[DIM];
    for(i = 0; i < dim; i++)
    {
        mean_th[i] = 0;
        raw2_th[i] = 0;
        raw3_th[i] = 0;
        raw4_th[i] = 0;
        for(j = 0; j < n; j++)
        {
            mean_th[i] += x[j + i * N];
            raw2_th[i] += x[j + i * N] * x[j + i * N];
            raw3_th[i] += x[j + i * N] * x[j + i * N] *x[j + i * N];
            raw4_th[i] += x[j + i * N] * x[j + i * N] * x[j + i * N] * x[j + i * N];
        }
        mean_th[i] /= N;
        raw2_th[i] /= N;
        raw3_th[i] /= N;
        raw4_th[i] /= N;

        if(fabs(mean_th[i] - mean[i]) > P_THRESHOLD) {
            errnums++;
        }
    }

    /***** Printing results *****/
    printf("Task dimension : %d\n", (int)dim);
    printf("Number of observations : %d\n\n", (int)n);

    // /***** Printing computed minimum, maximum, mean and moments estimates *****/
    printf("               Min        Max        Mean       2nd_raw   ");
    printf("  3rd_raw      4th_raw      2nd_cen    3rd_cen    4th_cen\n");
    for(i = 0; i < dim; i++)
    {
        printf("Variable #%i:  %+lf  %+lf  %+lf  %+lf  %+lf  %+lf", i + 1,
            min_estimate[i], max_estimate[i], mean[i],
            raw2[i], raw3[i], raw4[i]);
        printf("  %+lf  %+lf  %+lf\n", cen2[i], cen3[i], cen4[i]);
    }

    /***** Printing computed kurtosis, skewness and variation estimates *****/
    printf("\n               Kurtosis   Skewness   Variation\n");
    for(i = 0; i < dim; i++)
    {
        printf("Variable #%i:  %+lf  %+lf  %+lf\n", i + 1,
            kurtosis[i], skewness[i], variation[i]);
    }


    /***** Printing summary of the test *****/
    if (errnums == 0)
    {
        printf("\n\nAll the computed estimates agree with theory.\n");
    }
    else
    {
        printf("\n\nError: At least one of the computed estimates");
        printf(" disagrees with theory.\n");
        return 1;
    }

    /***** Delete Summary Statistics task *****/
    errcode = vslSSDeleteTask( &task );
    CheckVslError(errcode);

    mkl_free(x);
    mkl_free(min_estimate);
    mkl_free(max_estimate);
    mkl_free(mean);
    mkl_free(raw2);
    mkl_free(raw3);
    mkl_free(raw4);
    mkl_free(cen2);
    mkl_free(cen3);
    mkl_free(cen4);
    mkl_free(variation);
    mkl_free(skewness);
    mkl_free(kurtosis);

    return errcode;
}
