/*******************************************************************************
* Copyright (C) 2024 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*  Content:
*      cheevd OpenMP Offload Example
*******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <omp.h>
#include "mkl.h"
#include "mkl_omp_offload.h"

#ifdef MKL_ILP64
  #define FMT "%lld"
#else
  #define FMT "%d"
#endif


#define RAND() (2.0 * (float)rand() / (float)RAND_MAX - 1.0)

void init_rand(MKL_INT m, MKL_INT n, MKL_Complex8* a, MKL_INT lda) {
    for (MKL_INT j = 0; j < n; j++)
        for (MKL_INT i = 0; i < m; i++) {
            a[i + j * lda].real = RAND();
            a[i + j * lda].imag = RAND();
        }
}

void* alloc_device_default(size_t size) {
    void* ptr = omp_target_alloc(size, omp_get_default_device());
    if (!ptr) {
        printf("Error: couldn't allocate device memory\n");
        exit(1);
    }
    return ptr;
}

void free_device_default(void* ptr) {
    omp_target_free(ptr, omp_get_default_device());
}

int main() {
    // cheevd parameters
    char jobz = 'V';
    char uplo = 'L';
    MKL_INT n = 24;
    MKL_Complex8* a;
    MKL_INT lda = 33;
    float* w;
    MKL_Complex8* work;
    MKL_INT lwork = -1;
    float* rwork;
    MKL_INT lrwork = -1;
    MKL_INT* iwork;
    MKL_INT liwork = -1;
    MKL_INT info;
    // workspace query parameters
    MKL_Complex8 work_query;
    float rwork_query;
    MKL_INT iwork_query;

    printf("\nSolving Symmetric Eigenvalue Problem:\n");
    printf("===================================================================\n");
    printf("  n   = " FMT "\n", n);
    printf("  lda = " FMT "\n", lda);
    printf("===================================================================\n");

    // Allocate host memory for input/output arrays
    a = (MKL_Complex8*)mkl_malloc(lda * n * sizeof(MKL_Complex8), 64);
    w = (float*)mkl_malloc(n * sizeof(float), 64);

    // Initialize data for input arrays
    srand(1);
    init_rand(n, n, a, lda);
    for (MKL_INT i = 0; i < n; i++)
        a[i + i * lda].imag = (float)0;

    // Query workspace on device
#pragma omp target data map(from:work_query, rwork_query, iwork_query, info)
    {
#pragma omp dispatch
        cheevd(&jobz, &uplo, &n, a, &lda, w, &work_query, &lwork, &rwork_query, &lrwork, &iwork_query, &liwork, &info);
    }

    // Set work array sizes
    lwork = (MKL_INT)work_query.real;
    lrwork = (MKL_INT)rwork_query;
    liwork = (MKL_INT)iwork_query;

    // Allocate device workspace
    work = (MKL_Complex8*)alloc_device_default(lwork * sizeof(MKL_Complex8));
    rwork = (float*)alloc_device_default(lwork * sizeof(float));
    iwork = (MKL_INT*)alloc_device_default(liwork * sizeof(MKL_INT));

    // Solve SEP on device
#pragma omp target data map(tofrom:a[0:lda*n]) map(from:w[0:n], info)
    {
#pragma omp dispatch
        cheevd(&jobz, &uplo, &n, a, &lda, w, work, &lwork, rwork, &lrwork, iwork, &liwork, &info);
    }

    // Free device workspace
    free_device_default(work);
    free_device_default(rwork);
    free_device_default(iwork);

    // Examine output data
    if (info)
        printf("\nError: info = " FMT "\n", info);
    else
        printf("\nExample completed successfully\n");

    // Free host memory
    mkl_free(a);
    mkl_free(w);
}
