# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import List, Dict

import pandas as pd

from cli.writers.excel.excel_sheet import ExcelSheet, ExcelSheetFactory
from mpp import ViewAttributes, ViewType

@dataclass
class ExcelSheetInfo:
    file_path: str
    name: str
    tab_color: str


class BaseExcelWriter(ABC):

    def __init__(self, file_path: str, output_name: str, include_details=False, include_charts=False):
        self.file_path = file_path
        self.output_name = output_name
        self.include_details = include_details
        self.include_charts = include_charts

    @abstractmethod
    def write_csv_to_excel(self, views: List[ViewAttributes]) -> None:
        pass

    def _get_sorted_sheet_list(self, views: List[ViewAttributes]) -> List[ExcelSheetInfo]:
        sorted_excel_info = []
        for sheet in self.__sort_sheets(self.__get_sheet_map(views)):
            excel_info = ExcelSheetInfo(sheet.file_path, sheet.name, sheet.tab_color)
            sorted_excel_info.append(excel_info)
        return sorted_excel_info

    def __get_sheet_map(self, views: List[ViewAttributes]) -> Dict[str, ExcelSheet]:
        sheet_map = {}
        for view in views:
            sheet = ExcelSheetFactory.create(self.file_path, view, self.include_details, self.include_charts)
            sheet_map[sheet.name] = sheet
        return sheet_map

    def __sort_sheets(self, sheet_map: Dict[str, ExcelSheet]):
        sheet_order = {}
        for idx, (sheet_name, sheet) in enumerate(sheet_map.items()):
            sheet_order[sheet_name] = [idx]
            sheet_order.update(self.__get_sheet_order(sheet_order, sheet))
        sheet_order = pd.Series(sheet_order).sort_values()
        order = []
        for sheet in sheet_map:
            order.append(sheet_order[sheet])
        return [sheet_map[name] for name in sheet_order.index]

    @staticmethod
    def __get_sheet_order(sheet_order: Dict[str, List[int]], sheet: ExcelSheet) -> Dict:
        """
        Given a view, will provide the proper order in which the view should appear in the final Excel output,
        based on the following assumptions about order:
        1. Summary views (including per-txn)
        2. chart views
        3. detail views

        The ordering becomes a length 2 tuple, where the first index is the order in which the view is produced,
        and the second index is based on the assumptions listed above. This ensures, for example, that system sheets
        will come before socket sheets because system views are generated before socket views, and summary sheets
        come before detail sheets.
        """
        if sheet.type == ViewType.SUMMARY or sheet.type == ViewType.TPS_SUMMARY:
            sheet_order[sheet.name].insert(0, 1)
        elif 'chart' in sheet.name:
            sheet_order[sheet.name].insert(0, 2)
        elif sheet.type == ViewType.DETAILS:
            sheet_order[sheet.name].insert(0, 3)
        sheet_order[sheet.name] = tuple(sheet_order[sheet.name])
        return sheet_order

