# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import argparse
import os
import re
import sys
from datetime import datetime
from typing import List

from tools.common.validators.valid_input_types import ValidFileSpec
from tools.common.validators.input_validators import DirectoryValidator
from .format import DefaultFileIds

# Re-export ValidFileSpec so it's available from cli.args
__all__ = ['ValidFileSpec', 'ValidMultiFileSpec', 'ValidMultiInputFileSpec', 'ValidPathSpec', 
           'ValidInputPathSpec', 'ValidSample', 'FloatRange', 'IntegerRange', 'ValidOutputFormat',
           'TIMESTAMP_ARG_FORMAT', 'TIMESTAMP_ARG_FORMAT_MS']

# Timestamp formats for sample validation
TIMESTAMP_ARG_FORMAT = '%m/%d/%Y %H:%M:%S'
TIMESTAMP_ARG_FORMAT_MS = '%m/%d/%Y %H:%M:%S.%f'


class ValidMultiFileSpec:
    """
    ArgumentParser "type" validator for command line options that accept file names/paths and may include
    a prepended core type specifier. This class focuses purely on validation.
    """

    class MultiFileSpecError(Exception):
        def __init__(self, message):
            super().__init__(message)

    def __init__(self, file_must_exist=True, file_must_not_exist=False, max_file_size=0, allow_symlinks=True):
        self.__file_validator = ValidFileSpec(file_must_exist, file_must_not_exist, max_file_size, allow_symlinks)
        self.__file_id: str = DefaultFileIds.METRIC_CHART_FILE_PATH
        self.__file: str = ''

    @property
    def file_id(self):
        return self.__file_id

    @file_id.setter
    def file_id(self, file_id: str):
        self.__file_id = file_id

    def __call__(self, file_spec: str):
        """
        Validate a file specification string and return a dictionary format.
        
        :param file_spec: A string that represents a file with path. For hybrid files, the string is
                prepended with <core_type>=
                Non-hybrid file format: <file name with path>
                hybrid file format: <core_type>=<file name with path>
        :return: a single element dictionary where key:value is <core_type>:<file name with path>,
                 { <core_type>: <file name with path> }, for non-hybrid entries, the core_type is the
                 default core type.
        """
        self._parse_and_validate_file_spec(file_spec)
        return self._construct_validated_file_arg()

    def _construct_validated_file_arg(self):
        """Construct the validated file argument dictionary."""
        validated_file = self.__file_validator(self.__file)
        return {self.__file_id: validated_file}

    def _parse_and_validate_file_spec(self, file_spec: str):
        """
        Parse and validate the file specification for hybrid and non-hybrid entries
        :param file_spec: a single or hybrid core file specification
        """
        self.__file = file_spec
        hybrid_arg_separator: str = "="
        if hybrid_arg_separator in file_spec:
            args = file_spec.split(hybrid_arg_separator)
            ValidMultiFileSpec._validate_hybrid_args(file_spec, args)
            self.__file_id = args[0].lower()
            self.__file = args[1]
            self._validate_hybrid_core_type()

    @staticmethod
    def _validate_hybrid_args(file_spec: str, args: List[str]):
        """
        Sanity check a hybrid file specification
        It is expected to have exactly two non-empty args
        """
        expected_num_args = 2
        if '' in args or len(args) != expected_num_args:
            raise argparse.ArgumentTypeError(f'Invalid format: {file_spec}. '
                                             f'Expected: <core type>=<file name with path>.')

    def _validate_hybrid_core_type(self):
        """
        Sanity check the hybrid core type
        Hybrid cores cannot be the default core type
        Note: core type can't be fully validated until after the system information is parsed in main.
              see validate_multi_file_core_types for validating after both the command line and the data file
              have been parsed.
        """
        if self.__file_id == DefaultFileIds.METRIC_CHART_FILE_PATH:
            raise argparse.ArgumentTypeError(f'{self.__file_id} is not a valid hybrid core type.')

    @staticmethod
    def validate_multi_file_core_types(file_spec_dict, valid_core_types):
        """
        This public helper function is provided to check that core types entered on the command line
        are valid. It raises an error if the core_type from the command line arg is not documented in
        the data file unique/valid core types list.
        :param file_spec_dict: The dictionary, produced by the argument parser, for the file specification
                               arg to be validated.
                               (examples: args.metric_file_path and args.chart_format_file_path).
        :param valid_core_types: a list of known valid core types from the data file parser
        """
        from mpp.core.devices import DeviceType
        
        for core_type in file_spec_dict:
            if core_type not in valid_core_types and core_type != DefaultFileIds.DIRECTORY:
                if core_type == DeviceType.CORE:
                    raise argparse.ArgumentTypeError(f'When postprocessing data files for hybrid platforms, core type '
                                                     f'must be specified in the format <hybrid core type>=<metric definition file> ' 
                                                     f'for each metric file on the command line. '
                                                     f'Valid core types for the given data file are {valid_core_types}. '
                                                     f'Run \'mpp.py --help\' for more information on -m usage including auto-detection options.')
                else:
                    raise argparse.ArgumentTypeError(f'{core_type} is not a supported core type. '
                                                     f'Valid core types for the specified data file are {valid_core_types}')


class ValidMultiInputFileSpec(ValidMultiFileSpec):

    def __init__(self, file_must_exist=True, file_must_not_exist=False, max_file_size=0, allow_symlinks=True):
        super().__init__(file_must_exist, file_must_not_exist, max_file_size, allow_symlinks)
        self.file_id = DefaultFileIds.INPUT_DATA_FILE_PATH


class ValidPathSpec:

    def __init__(self, file_must_exist=True, file_must_not_exist=False, max_file_size=0, allow_symlinks=True):
        self.__file_must_exist = file_must_exist
        self.__file_must_not_exist = file_must_not_exist
        self.__max_file_size = max_file_size
        self.__allow_symlinks = allow_symlinks

    @property
    def file_must_exist(self):
        return self.__file_must_exist

    @property
    def file_must_not_exist(self):
        return self.__file_must_not_exist

    @property
    def max_file_size(self):
        return self.__max_file_size

    @property
    def allow_symlinks(self):
        return self.__allow_symlinks

    def __call__(self, path_spec: str):
        if os.path.isdir(path_spec):
            return {DefaultFileIds.DIRECTORY: DirectoryValidator().__call__(path_spec)}
        return ValidMultiFileSpec(self.__file_must_exist, self.__file_must_not_exist, self.__max_file_size,
                                  self.__allow_symlinks).__call__(path_spec)


class ValidInputPathSpec(ValidPathSpec):

    def __call__(self, path_spec: str):
        if os.path.isdir(path_spec):
            return {DefaultFileIds.DIRECTORY: DirectoryValidator().__call__(path_spec)}
        return ValidMultiInputFileSpec(self.file_must_exist, self.file_must_not_exist, self.max_file_size,
                                  self.allow_symlinks).__call__(path_spec)


class IntegerRangeListErrorMessages:
    INVALID_PATTERN = '{}: must be a comma-separated list of integers or ranges'
    INVALID_MIN_MAX = '{}: value must be an integer between {} and {}'
    INVALID_RANGE = '{}: range start must not be greater than end'


class ValidIntegerRangeList:
    """ArgumentParser "type" validator for command line options that accept a list of integers"""

    VALID_PATTERN = re.compile(r'^(?:\d+|\d+-\d+)(?:,(?:\d+|\d+-\d+))*$')

    def __init__(self, min_value=0, max_value=sys.maxsize):
        """
        Initialize the validator with optional min and max values.

        :param min_value: Minimum value for the integer range (default is 0)
        :param max_value: Maximum value for the integer range (default is sys.maxsize)
        """
        self.__min_value = min_value
        self.__max_value = max_value

    def __call__(self, value):
        """
        Validate the input value as a comma-separated list of integers or ranges.

        :param value: A string representing a list of integers or ranges (e.g., "1,2,3" or "1-5,10")
        :return: A list of integers if valid
        :raises argparse.ArgumentTypeError: If the input is not valid
        """
        if value is None:
            return
        self._validate_argument_pattern(value)
        self._validate_min_max_range(value)
        self._validate_increasing_ranges(value)
        return value

    def _validate_argument_pattern(self, value):
        if not self.VALID_PATTERN.match(value):
            raise argparse.ArgumentTypeError(IntegerRangeListErrorMessages.INVALID_PATTERN.format(value))

    def _validate_min_max_range(self, value):
        numbers = re.findall(r'\d+', value)
        for number in numbers:
            number_as_int = int(number)
            if not (self.__min_value <= number_as_int <= self.__max_value):
                raise argparse.ArgumentTypeError(IntegerRangeListErrorMessages.INVALID_MIN_MAX
                                                     .format(number, self.__min_value, self.__max_value))

    @staticmethod
    def _validate_increasing_ranges(value):
        for part in value.split(','):
            if '-' in part:
                start, end = map(int, part.split('-'))
                if start > end:
                    raise argparse.ArgumentTypeError(IntegerRangeListErrorMessages.INVALID_RANGE.format(part))


class ValidSample:
    """
    ArgumentParser "type" validator for command line options that accept sample number or timestamp
    """

    def __call__(self, sample_reference):
        try:
            sample_as_int = int(sample_reference)
            if '.' in str(sample_reference) or sample_as_int < 1:
                raise argparse.ArgumentTypeError(f'{sample_reference}: '
                                                 f'sample number must be equal to or greater than 1')
            return sample_as_int
        except ValueError:
            try:
                return datetime.strptime(sample_reference, TIMESTAMP_ARG_FORMAT_MS)
            except ValueError:
                try:
                    return datetime.strptime(sample_reference, TIMESTAMP_ARG_FORMAT)
                except ValueError:
                    raise argparse.ArgumentTypeError(f'{sample_reference}: not a valid timestamp or sample number')


class FloatRange:
    """
    ArgumentParser "type" validator for numeric command line options with optional min/max values
    """

    def __init__(self, max_value=float(sys.maxsize)):
        self.__max_value = max_value

    def __call__(self, value):
        if 'e' in str(value) or 'E' in str(value):
            raise argparse.ArgumentTypeError(f'{value}: value must be a valid float number')

        try:
            value_as_float = float(value)
            if not (0 < value_as_float <= self.__max_value):
                raise argparse.ArgumentTypeError(f'{value}: value must be a number greater than '
                                                 f'0 and less than {self.__max_value}')
            return value_as_float
        except ValueError:
            raise argparse.ArgumentTypeError(f'{value}: value must be a valid float number')


class IntegerRange:
    """
    ArgumentParser "type" validator for numeric command line options with optional min/max values
    """

    def __init__(self, min_value=0, max_value=sys.maxsize):
        self.__min_value = min_value
        self.__max_value = max_value

    def __call__(self, value):
        if '.' in str(value):
            raise argparse.ArgumentTypeError(f'{value}: value must be an integral (whole) number')

        try:
            value_as_int = int(value)
            if not (self.__min_value <= value_as_int <= self.__max_value):
                raise argparse.ArgumentTypeError(f'{value}: value must be an integral (whole) number between '
                                                 f'{self.__min_value} and {self.__max_value}')
            return value_as_int
        except ValueError:
            raise argparse.ArgumentTypeError(f'{value}: value must be an integral (whole) number')


class ValidOutputFormat:

    XLSX = 'xlsx'

    def __init__(self):
        self.__valid_output_formats = [self.XLSX]

    def __call__(self, output_format):
        if output_format not in self.__valid_output_formats:
            raise argparse.ArgumentTypeError(f'{output_format}: valid format options: {self.__valid_output_formats}')
        return output_format
