# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

import argparse

from tools.common.validators.valid_input_types import ValidString, ValidPathNameAndFileSpec
from tools.common.virtual_environment import VirtualEnvironment, VENV_args


class ArgParserException(Exception):
    def __init__(self, message):
        super().__init__(message)


class CommonArgParser(argparse.ArgumentParser):
    DEFAULT_DESCRIPTION = 'Common development and test script command line options'

    def __init__(self, description = DEFAULT_DESCRIPTION, add_help=True):
        super().__init__(description=description, add_help=add_help)

    def parse_args(self, args=None, namespace=None):
        args, unknown = self.parse_known_args(args, namespace)
        if unknown:
            self.print_help()
            raise ArgParserException(f'Unknown arguments: {" ".join(unknown)}')
        self.__add_venv_args_attribute(args)
        return args

    def add_logging_args(self):
        logging_options = self.add_argument_group('Logging Options')
        logging_options.add_argument(
            '--verbose',
            action='store_true',
            help='increase output level to provide more information')
        logging_options.add_argument(
            '--log-level',
            default='WARNING',
            choices=['DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL'],
            type=ValidString(),
            help='set logging level')

    def add_venv_args(self):
        venv_options = self.add_argument_group('Virtual Environment Options')
        venv_options.add_argument(
            '--venv-path',
            default=VirtualEnvironment.DEFAULT_VENV,
            type=ValidPathNameAndFileSpec(file_must_exist=False, allow_symlinks=False),
            help='virtual environment location. Default is .venv in the current directory')
        venv_options.add_argument(
            '--venv-clean',
            default=False,
            action='store_true',
            help='force the command to remove and recreate the virtual environment')
        venv_options.add_argument(
            '--venv-ignore',
            default=False,
            action='store_true',
            help=argparse.SUPPRESS)

    @staticmethod
    def __add_venv_args_attribute(args):
        venv_args = VENV_args()
        if hasattr(args, 'venv_path'):
            venv_args.path_str = args.venv_path
        if hasattr(args, 'venv_clean'):
            venv_args.clean = args.venv_clean
        if hasattr(args, 'verbose'):
            venv_args.verbose = args.verbose
        if hasattr(args, 'venv_ignore') and args.venv_ignore:
            venv_args = None
        args.venv_args = venv_args

