# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

import re
from dataclasses import dataclass
from pathlib import Path
from typing import Pattern

import numpy as np
import pytz

from mpp.parsers.data_parser import DataParser, ParserAttributes, _Line, EventPatterns
from mpp.parsers.pcm.pcm_system_information import PcmSystemInformationParser


@dataclass
class PcmEventPatterns(EventPatterns):
    STATIC_SCOPED_RE: Pattern = re.compile(r'REG_STATIC_([A-Z]+)_|STATIC:.*scope=(.*)')


class _PcmLine(_Line):
    """
    Store a line from the PCM file, extracting the relevant information as class attributes.

    Extract attributes from a line in the PCM raw file.  Each line represents the data about a given event for
    the sample duration.  Each line may have a breakdown of the event counts of the system across
    sockets/cores/threads/channels/etc.

    """

    def __init__(self, line_values, unique_os_processors, timestamp):
        super().__init__(line_values, unique_os_processors, timestamp)
        # Attributes must be retrieved in this order for PcmParser, because we use the pop method: name, device,
        # tsc_count, values
        self._timestamp = timestamp
        self._name = self.line_values.pop(2)
        self.line_values.pop(0)
        self.line_values.pop(0)
        self.line_values.pop(0)
        self._event_patterns = PcmEventPatterns()
        self._device = self._set_event_device()
        self._tsc_count = float(self.line_values.pop(0))
        self._values = np.array([v.replace(',', '').replace('N/A', 'nan') for v in self.line_values], dtype=float)


class PcmParser(DataParser):
    from mpp.core.types import CollectorIds

    attributes = ParserAttributes(',', "Update every", '%Y-%m-%d,%H:%M:%S.%f',
                                  re.compile(r'(\d{4})-(\d{2})-(\d{2}),(\d{2}):(\d{2}):(\d{2}).(\d{3})'))

    line_class = _PcmLine

    collector = CollectorIds.PCM

    def __init__(self,
                 input_file: Path,
                 timezone: pytz.tzinfo = None,
                 ref_tsc_hz: int = 0,
                 **kwargs):
        """
        Initialize the PCM parser

        :param input_file: the PCM data file to parse
        :param timezone: an optional timezone object for converting timestamp strings
        """
        # super().__init__ requires initialized system information
        self._system_info = PcmSystemInformationParser(input_file, ref_tsc_hz)
        super().__init__(input_file, timezone)

    def _get_timestamps_not_found_error_class(self):
        return _PcmTimeStampsNotFoundError

    def convert_to_datetime_from_line(self, line: str):
        values = line.split(',')
        return self.convert_to_datetime(values[0] + ',' + values[1])

    @property
    def system_info(self):
        return self._system_info


class _PcmTimeStampsNotFoundError(Exception):
    def __init__(self, message="No samples labeled with timestamps found."):
        super().__init__(message)

