/*******************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API oneapi::mkl::sparse::sort_matrix to perform general
*       sparse matrix sorting on a SYCL device (CPU, GPU). This example uses a
*       sparse matrix in CSR format.
*
*       The input matrix is sorted in-place and no temporary workspace is used.
*
*       The supported floating point data types for sort_matrix matrix data are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for sort_matrix are:
*           CSR
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix sorting
//
template <typename dataType, typename intType>
int run_sparse_blas_example(sycl::queue &q)
{

    bool good = true;

    // handle for sparse matrix
    oneapi::mkl::sparse::matrix_handle_t csrA = nullptr;

    //
    // array memory management tools
    //
    std::vector<intType *> int_ptr_vec;
    std::vector<dataType *> data_ptr_vec;
    std::vector<std::int64_t *> i64_ptr_vec;
    std::vector<void *> void_ptr_vec;

    try {

        // Initialize data for Sparse Matrix
        oneapi::mkl::index_base a_index = oneapi::mkl::index_base::one;
        intType size = 4;
        const std::int64_t a_nrows = size * size * size;
        const std::int64_t a_ncols = a_nrows;
              std::int64_t a_nnz   = 27 * a_nrows;

        //
        // setup A data locally in CSR format
        //
        std::vector<intType, mkl_allocator<intType, 64>> ia;
        std::vector<intType, mkl_allocator<intType, 64>> ja;
        std::vector<dataType, mkl_allocator<dataType, 64>> a;

        ia.resize(a_nrows + 1);
        ja.resize(27 * a_nrows);
        a.resize(27 * a_nrows);

        intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
        generate_sparse_matrix<dataType, intType>(size, ia, ja, a, a_ind);
        a_nnz = ia[a_nrows] - a_ind;
        shuffle_matrix_data<dataType, intType>(ia, ja, a, a_ind, a_nrows, a_nnz);

        intType *a_rowptr  = sycl::malloc_shared<intType>(a_nrows + 1, q);
        intType *a_colind  = sycl::malloc_shared<intType>(a_nnz, q);
        dataType *a_values = sycl::malloc_shared<dataType>(a_nnz, q);
        if (!a_rowptr || !a_colind || !a_values) {
            std::string errorMessage =
                "Failed to allocate USM shared memory arrays \n"
                " for CSR A matrix: a_rowptr(" + std::to_string((a_nrows+1)*sizeof(intType)) + " bytes)\n"
                "                   a_colind(" + std::to_string((a_nnz)*sizeof(intType)) + " bytes)\n"
                "                   a_values(" + std::to_string((a_nnz)*sizeof(dataType)) + " bytes)";

            throw std::runtime_error(errorMessage);
        }
        int_ptr_vec.push_back(a_rowptr);
        int_ptr_vec.push_back(a_colind);
        data_ptr_vec.push_back(a_values);

        // copy data to USM arrays
        for (intType i = 0; i < a_nrows + 1; ++i) {
            a_rowptr[i] = ia[i];
        }
        for (intType i = 0; i < a_nnz; ++i) {
            a_colind[i] = ja[i];
            a_values[i]    = a[i];
        }

        //
        // compute some stats about sparse matrix
        //
        const double ave_nnz_per_row = static_cast<double>(a_nnz)/a_nrows;

        intType max_nnz_per_row = 0;
        intType min_nnz_per_row = a_nnz;
        for (intType row = 0; row < a_nrows; ++row) {
            const intType loc_nnz_per_row = a_rowptr[row+1] - a_rowptr[row];

            if (loc_nnz_per_row > max_nnz_per_row)
                max_nnz_per_row = loc_nnz_per_row;

            if (loc_nnz_per_row < min_nnz_per_row)
                min_nnz_per_row = loc_nnz_per_row;
        }

        //
        // Execute Matrix Sort
        //

        std::cout << "\n\t\tsparse::sort_matrix parameters:\n";
        std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
        std::cout << "\t\t\tA_index = " << a_index << std::endl;
        std::cout << "\t\t\tA_nnz   = " << a_nnz << std::endl;
        std::cout << "\t\t\t\tmin_nnz_per_row = " << min_nnz_per_row << std::endl;
        std::cout << "\t\t\t\tave_nnz_per_row = " << ave_nnz_per_row << std::endl;
        std::cout << "\t\t\t\tmax_nnz_per_row = " << max_nnz_per_row << std::endl;


        // print out first bit of unsorted A matrix before sorting
        sycl::event ev_input = q.submit([&](sycl::handler &cgh) {
            auto kernel = [=]() {
                std::cout << "[INPUT] Unsorted A matrix [first two rows]:" << std::endl;
                intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
                for (intType row = 0; row < std::min(static_cast<std::int64_t>(2), a_nrows); ++row) {
                    for (intType j = a_rowptr[row] - a_ind; j < a_rowptr[row + 1] - a_ind; ++j) {
                        intType col = a_colind[j];
                        dataType val  = a_values[j];
                        std::cout << "C(" << row + a_ind << ", " << col << ") = " << val
                                  << std::endl;
                    }
                }
            };
            cgh.host_task(kernel);
        });

        oneapi::mkl::sparse::init_matrix_handle(&csrA);

        auto ev_set = oneapi::mkl::sparse::set_csr_data(q, csrA, a_nrows, a_ncols, a_nnz, a_index, a_rowptr, a_colind, a_values, {});

        // Sort the sparse matrix
        auto ev_sort = oneapi::mkl::sparse::sort_matrix(q, csrA, {ev_set, ev_input});

        auto ev_release = oneapi::mkl::sparse::release_matrix_handle(q, &csrA, {ev_sort});

        ev_release.wait();

        //
        // Post Processing
        //

        // print out first bit of sorted matrix A
        sycl::event ev_print = q.submit([&](sycl::handler &cgh) {
            cgh.depends_on({ev_sort});
            auto kernel = [=]() {
                std::cout << "[OUTPUT] Sorted A matrix [first two rows]:" << std::endl;
                intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
                for (intType row = 0; row < std::min(static_cast<std::int64_t>(2), a_nrows); ++row) {
                    for (intType j = a_rowptr[row] - a_ind; j < a_rowptr[row + 1] - a_ind; ++j) {
                        intType col = a_colind[j];
                        dataType val = a_values[j];
                        std::cout << "C(" << row + a_ind << ", " << col << ") = " << val
                                  << std::endl;
                    }
                }
            };
            cgh.host_task(kernel);
        });
        ev_print.wait();

        good = true;

        q.wait_and_throw();

    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;
        good = false;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;
        good = false;
    }

    q.wait();

    // backup cleaning of matrix handle and others for if exceptions happened
    if (csrA) oneapi::mkl::sparse::release_matrix_handle(q, &csrA).wait();

    cleanup_arrays<dataType, intType>(data_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, q);

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# CSR Matrix Sorting Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::sort_matrix" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type,
//  run_sparse_blas_example is run with all supported data types,
//  if any fail, we move on to the next device.
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {
        try {
            sycl::device my_dev;
            bool my_dev_is_found = false;
            get_sycl_device(my_dev, my_dev_is_found, *it);

            if (my_dev_is_found) {
                std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

                // Catch asynchronous exceptions
                auto exception_handler = [](sycl::exception_list exceptions) {
                    for (std::exception_ptr const &e : exceptions) {
                        try {
                            std::rethrow_exception(e);
                        }
                        catch (sycl::exception const &e) {
                            std::cout << "Caught asynchronous SYCL exception: \n"
                                << e.what() << std::endl;
                        }
                    }
                };

                sycl::queue q(my_dev, exception_handler);

                std::cout << "\tRunning with single precision real data type:" << std::endl;
                status |= run_sparse_blas_example<float, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision real data type:" << std::endl;
                    status |= run_sparse_blas_example<double, std::int32_t>(q);
                }

                std::cout << "\tRunning with single precision complex data type:" << std::endl;
                status |= run_sparse_blas_example<std::complex<float>, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision complex data type:" << std::endl;
                    status |= run_sparse_blas_example<std::complex<double>, std::int32_t>(q);
                }

            }
            else {
#ifdef FAIL_ON_MISSING_DEVICES
                std::cout << "No " << sycl_device_names[*it]
                    << " devices found; Fail on missing devices "
                    "is enabled.\n";
                return 1;
#else
                std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                    << sycl_device_names[*it] << " tests.\n";
#endif
            }
        }
        catch (sycl::exception const &e) {
            std::cout << "\t\tCaught SYCL exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }
        catch (std::exception const &e) {
            std::cout << "\t\tCaught std exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }


    } // for loop over devices

    mkl_free_buffers();
    return status;
}
