/*******************************************************************************
* Copyright (C) 2022 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Sparse BLAS C OpenMP
*             offload example for mkl_sparse_order with synchronous execution.
*
********************************************************************************
*
* Consider the matrix A (see 'Sparse Storage Formats for Sparse BLAS Level 2
* and Level 3 in the  Intel oneMKL Reference Manual')
*
*                 |   1       -1      0   -3     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*  The matrix A is represented in a zero-based compressed sparse row (CSR) storage
*  scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
*  Intel oneMKL Reference Manual) with arrays sorted by the column
*  indices over each row as follows:
*
*         values    = ( 1 -1 -3 -2  5  4  6  4 -4  2  7  8 -5 )
*         columns   = ( 0  1  3  0  1  2  3  4  0  2  3  1  4 )
*         row_index = ( 0        3     5        8       11    13 )
*
*  One of the many unsorted CSR representations of the above matrix may be as
*  follows:
*
*         values    = (-1  1 -3  5 -2  4  4  6  2  7 -4  8 -5 )
*         columns   = ( 1  0  3  1  0  4  2  3  2  3  0  1  4 )
*         row_index = ( 0        3     5        8       11    13 )
*
*  Note that `columns` and `values` arrays above are not sorted over each row;
*  the CSR format by definition has its `row_index` array sorted.
*
*  This test performs the following operations :
*
*       Sort the `columns` and `values` arrays of the CSR matrix using
*       mkl_sparse_order omp offload.
*
********************************************************************************
*/
#include <assert.h>
#include <omp.h>
#include <stdio.h>
#include <stdlib.h>

#include "common_for_sparse_examples.h"
#include "mkl.h"
#include "mkl_omp_offload.h"

// special formatting for MKL_INT and floating point printing below
// with at least 3 digits and accounting for aligned space for negatives
#ifdef MKL_ILP64
#define INT_FORMAT "% 3lld "
#else
#define INT_FORMAT "% 3d "
#endif

#define FP_FORMAT "% 3.1f "


int main()
{
//*******************************************************************************
//     Declaration and initialization of parameters for sparse representation of
//     the matrix A in the compressed sparse row format:
//*******************************************************************************
#define M     5 // nRows of A
#define N     5 // nCols of A
#define NNZ   13
#define ALIGN 64

    //*******************************************************************************
    //    Declaration of local variables:
    //*******************************************************************************
    // // Structures with sparse matrices stored in CSR format
    sparse_matrix_t csrA = NULL;

    sparse_index_base_t indexing = SPARSE_INDEX_BASE_ZERO;

    MKL_INT i;

    float *values_sorted      = (float *)mkl_malloc(sizeof(float) * NNZ, ALIGN);
    MKL_INT *columns_sorted   = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * NNZ, ALIGN);
    MKL_INT *row_index_sorted = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * (M + 1), ALIGN);

    float *values_A      = (float *)mkl_malloc(sizeof(float) * NNZ, ALIGN);
    MKL_INT *columns_A   = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * NNZ, ALIGN);
    MKL_INT *row_index_A = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * (M + 1), ALIGN);

    const int num_pointers = 6;
    void *pointer_array[num_pointers];
    pointer_array[0] = values_sorted;
    pointer_array[1] = columns_sorted;
    pointer_array[2] = row_index_sorted;
    pointer_array[3] = values_A;
    pointer_array[4] = columns_A;
    pointer_array[5] = row_index_A;

    if (!values_sorted || !columns_sorted || !row_index_sorted ||
        !values_A      || !columns_A      || !row_index_A
       ) {
        free_allocated_memories(pointer_array, num_pointers);
        return 1;
    }

    //*******************************************************************************
    //    Sparse representation of the unsorted matrix `csrA` which gets sorted
    //*******************************************************************************
    {
        float init_values_sorted[NNZ]        = { 1,-1,-3,-2, 5, 4, 6, 4,-4, 2, 7, 8,-5};
        MKL_INT init_columns_sorted[NNZ]     = { 0, 1, 3, 0, 1, 2, 3, 4, 0, 2, 3, 1, 4};
        MKL_INT init_row_index_sorted[M + 1] = { 0,       3,    5,       8,      11,   13};
        float init_values_A[NNZ]             = {-1, 1,-3, 5,-2, 4, 4, 6, 2, 7,-4, 8,-5};
        MKL_INT init_columns_A[NNZ]          = { 1, 0, 3, 1, 0, 4, 2, 3, 2, 3, 0, 1, 4};
        MKL_INT init_row_index_A[M + 1]      = { 0,       3,    5,       8,      11,   13};

        for (i = 0; i < NNZ; i++) {
            values_sorted[i]  = init_values_sorted[i];
            columns_sorted[i] = init_columns_sorted[i];
            values_A[i]  = init_values_A[i];
            columns_A[i] = init_columns_A[i];
        }
        for (i = 0; i < M + 1; i++) {
            row_index_sorted[i] = init_row_index_sorted[i];
            row_index_A[i] = init_row_index_A[i];
        }

    }

    printf("\n EXAMPLE PROGRAM FOR mkl_sparse_order omp_offload\n");
    printf("---------------------------------------------------\n");
    printf("\n");
    printf("   INPUT DATA FOR mkl_sparse_order omp offload\n");
    printf("   WITH GENERAL SPARSE MATRICES\n");
    printf("   SORTED csr:\n");
    printf("   row_index_sorted:\n        ");
    for (i = 0; i < M+1; i++) printf(INT_FORMAT, row_index_sorted[i]);
    printf("\n   columns_sorted:\n        ");
    for (i = 0; i < NNZ; i++) printf(INT_FORMAT, columns_sorted[i]);
    printf("\n   values_sorted:\n        ");
    for (i = 0; i < NNZ; i++) printf(FP_FORMAT, values_sorted[i]);
    printf("\n   UNSORTED csrA:                 \n");
    printf("   row_index_A:\n        ");
    for (i = 0; i < M+1; i++) printf(INT_FORMAT, row_index_A[i]);
    printf("\n   columns_A:\n        ");
    for (i = 0; i < NNZ; i++) printf(INT_FORMAT, columns_A[i]);
    printf("\n   values_A:\n        ");
    for (i = 0; i < NNZ; i++) printf(FP_FORMAT, values_A[i]);
    printf("\n---------------------------------------------------\n");
    fflush(stdout);

    sparse_status_t ie_status;

    // Create handle with matrix stored in CSR format
    printf("Create CSR matrix\n");
    ie_status = mkl_sparse_s_create_csr(&csrA, indexing,
                                        M, // number of rows
                                        N, // number of cols
                                        row_index_sorted, row_index_sorted + 1,
                                        columns_sorted, values_sorted);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_create_csr: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Sort column indices and values
    printf("Call mkl_sparse_order()\n");
    ie_status = mkl_sparse_order(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_order: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Release matrix handle and deallocate matrix
    printf("Destroy csr matrix\n");
    ie_status = mkl_sparse_destroy(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_destroy: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_order()\n");
    printf("   row_index_sorted:\n        ");
    for (i = 0; i < M+1; i++) printf(INT_FORMAT, row_index_sorted[i]);
    printf("\n   columns_sorted:\n        ");
    for (i = 0; i < NNZ; i++) printf(INT_FORMAT, columns_sorted[i]);
    printf("\n   values_sorted:\n        ");
    for (i = 0; i < NNZ; i++) printf(FP_FORMAT, values_sorted[i]);
    printf("\n---------------------------------------------------\n");
    fflush(stdout);

    const int devNum = 0;

    sparse_matrix_t csrA_gpu = NULL;

    sparse_status_t status_create_A, status_order_A, status_destroy_A;

// call create_csr/order/destroy via omp_offload.
#pragma omp target data map(to: row_index_A[0:M+1]) \
                        map(tofrom: columns_A[0:NNZ], values_A[0:NNZ]) \
                        device(devNum)
    {
        printf("Create CSR matrix via omp_offload\n");
        fflush(stdout);

#pragma omp dispatch device(devNum)
        status_create_A = mkl_sparse_s_create_csr(&csrA_gpu, indexing, M, N,
                                                  row_index_A, row_index_A + 1, columns_A, values_A);

        printf("Compute mkl_sparse_order via omp_offload\n");
        fflush(stdout);

#pragma omp dispatch device(devNum)
        status_order_A = mkl_sparse_order(csrA_gpu);

        printf("Destroy the CSR matrix via omp_offload\n");

#pragma omp dispatch device(devNum)
        status_destroy_A = mkl_sparse_destroy(csrA_gpu);
    }

    int flps_per_value = 0;
    int status1 = 0, status2 = 0, status3 = 0;

    int status_offload = status_create_A | status_order_A | status_destroy_A;
    if (status_offload != 0) {
        printf("\tERROR: status_create_A = %d, status_order_A = %d, status_destroy_A = %d\n",
               status_create_A, status_order_A, status_destroy_A);
        goto cleanup;
    }

    printf("   OUTPUT DATA FOR csrA mkl_sparse_order_omp_offload async execution.\n");
    printf("   sorted row_index_A:\n        ");
    for (i = 0; i < M+1; i++) printf(INT_FORMAT, row_index_A[i]);
    printf("\n   sorted columns_A:\n        ");
    for (i = 0; i < NNZ; i++) printf(INT_FORMAT, columns_A[i]);
    printf("\n   sorted values_A:\n        ");
    for (i = 0; i < NNZ; i++) printf(FP_FORMAT, values_A[i]);
    printf("\n---------------------------------------------------\n");
    fflush(stdout);

    status1 = validation_result_integer(row_index_sorted, row_index_A, M+1);
    status2 = validation_result_integer(columns_sorted, columns_A, NNZ);
    status3 = validation_result_float(values_sorted, values_A, NNZ, flps_per_value);

cleanup:
    free_allocated_memories(pointer_array, num_pointers);

    const int status_all = status1 | status2 | status3 |
                           status_offload;
    printf("Test %s\n", status_all == 0 ? "PASSED" : "FAILED");
    fflush(stdout);

    return status_all;
}
